/**
Copyright (C) 2025 Digital Venture Consultants (info@dvc.ventures)

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package ventures.dvc.karbon.provider.webclient

import arrow.core.nonFatalOrThrow
import org.springframework.http.MediaType
import org.springframework.web.reactive.function.client.ClientResponse
import org.springframework.web.reactive.function.client.WebClient
import org.springframework.web.reactive.function.client.awaitExchange
import ventures.dvc.karbon.location.ForecastLocation
import ventures.dvc.karbon.provider.*

internal class WebClientValidateKeyOperation(
    private val configuration: ForecastProviderConfiguration,
    private val client: WebClient,
) : ValidateKeyProvider {
    override suspend fun invoke(request: ValidateKeyRequest): ValidateKeyResponse {
        return try {
            validateKey(request)
        } catch (t: Throwable) {
            t.nonFatalOrThrow()
            ValidateKeyResponse.None(request.key.location)
        }
    }

    private suspend fun validateKey(request: ValidateKeyRequest): ValidateKeyResponse {
        val location = request.key.location
        val hash = request.key.hash

        return client.head()
            .uri(configuration.forecastsUrl(location))
            .accept(MediaType.APPLICATION_JSON)
            .headers { headers ->
                headers.ifNoneMatch(hash)
            }
            .awaitExchange { response ->
                when {
                    response.isOk() -> response.toModifiedOrNone(location)
                    response.isNotModified() -> ValidateKeyResponse.Valid(location, hash)
                    else -> ValidateKeyResponse.None(location)
                }
            }
    }
}

private fun ClientResponse.toModifiedOrNone(location: ForecastLocation) =
    when (val hash = forecastHash()) {
        is ForecastHash.Hash -> ValidateKeyResponse.Modified(location, hash)
        ForecastHash.None -> ValidateKeyResponse.None(location)
    }