/**
Copyright (C) 2025 Digital Venture Consultants (info@dvc.ventures)

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package ventures.dvc.karbon.provider.webclient

import arrow.core.Either
import arrow.core.left
import arrow.core.nonFatalOrThrow
import arrow.core.right
import org.springframework.http.MediaType
import org.springframework.web.reactive.function.client.ClientResponse
import org.springframework.web.reactive.function.client.WebClient
import org.springframework.web.reactive.function.client.awaitBody
import org.springframework.web.reactive.function.client.awaitExchange
import ventures.dvc.karbon.location.ForecastLocation
import ventures.dvc.karbon.provider.*
import ventures.dvc.karbon.serialization.KarbonForecastDeserializer

internal class WebClientForecastOperation(
    private val configuration: ForecastProviderConfiguration,
    private val serializer: KarbonForecastDeserializer,
    private val client: WebClient,
) : KarbonForecastProvider {
    override suspend fun invoke(request: ForecastRequest): Either<ForecastError, ForecastResponse> {
        return try {
            fetchForecast(request.location)
        } catch (t: Throwable) {
            t.nonFatalOrThrow()
            ForecastError.Generic(t).left()
        }
    }

    private suspend fun fetchForecast(location: ForecastLocation): Either<ForecastError, ForecastResponse> {
        return client.get()
            .uri(configuration.forecastsUrl(location))
            .accept(MediaType.APPLICATION_JSON)
            .awaitExchange { response ->
                when {
                    response.isOk() -> response.deserializeToForecastResponse(location).right()
                    response.isNotFound() -> ForecastError.LocationNotFound(location).left()
                    else -> ForecastError.NotAvailable(location, response.statusCode().toString()).left()
                }
            }
    }

    private suspend fun ClientResponse.deserializeToForecastResponse(
        location: ForecastLocation,
    ): ForecastResponse {
        val content = serializer { awaitBody<String>() }
        return ForecastResponse(
            key = ForecastKey(location = location, hash = ForecastHash.None),
            emissions = content.emissions
        )
    }
}