/**
 * Copyright (c) 2018-present, A2 Rešitve d.o.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See
 * the License for the specific language governing permissions and limitations under the License.
 */

package solutions.a2.oracle.jdbc;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;

import oracle.jdbc.OracleConnection;
import oracle.jdbc.OracleResultSet;
import oracle.jdbc.OracleStatement;
import oracle.jdbc.dcn.DatabaseChangeRegistration;

/**
 * 
 * Oracle JDBC Proxy Statement implementation
 * @author <a href="mailto:averemee@a2.solutions">Aleksei Veremeev</a>
 * 
 */
public class OraProxyStatement implements OracleStatement {

	final OracleConnection connection;
	final OracleStatement oracle;
	final OraProxySqlTranslator translator;

	OraProxyStatement(OracleConnection connection, OracleStatement statement, OraProxySqlTranslator translator) {
		this.connection = connection;
		this.oracle = statement;
		this.translator = translator;
	}

	@Override
	public ResultSet executeQuery(String sql) throws SQLException {
		return new OraProxyResultSet(this, (OracleResultSet) oracle.executeQuery(translator.translate(sql)));
	}

	@Override
	public int executeUpdate(String sql) throws SQLException {
		return oracle.executeUpdate(translator.translate(sql));
	}

	@Override
	public boolean execute(String sql) throws SQLException {
		return oracle.execute(translator.translate(sql));
	}

	@Override
	public Connection getConnection() throws SQLException {
		return connection;
	}

	@Override
	public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
		return oracle.executeUpdate(translator.translate(sql), autoGeneratedKeys);
	}

	@Override
	public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
		return oracle.executeUpdate(translator.translate(sql), columnIndexes);
	}

	@Override
	public int executeUpdate(String sql, String[] columnNames) throws SQLException {
		return oracle.executeUpdate(translator.translate(sql), columnNames);
	}

	@Override
	public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
		return oracle.execute(translator.translate(sql), autoGeneratedKeys);
	}

	@Override
	public boolean execute(String sql, int[] columnIndexes) throws SQLException {
		return oracle.execute(translator.translate(sql), columnIndexes);
	}

	@Override
	public boolean execute(String sql, String[] columnNames) throws SQLException {
		return oracle.execute(translator.translate(sql), columnNames);
	}

	@Override
	public ResultSet getResultSet() throws SQLException {
		return new OraProxyResultSet(this, (OracleResultSet) oracle.getResultSet());
	}

	@Override
	public ResultSet getGeneratedKeys() throws SQLException {
		return new OraProxyResultSet(this, (OracleResultSet) oracle.getGeneratedKeys());
	}

	@Override
	public <T> T unwrap(Class<T> iface) throws SQLException {
		return oracle.unwrap(iface);
	}

	@Override
	public boolean isWrapperFor(Class<?> iface) throws SQLException {
		return oracle.isWrapperFor(iface);
	}

	@Override
	public void close() throws SQLException {
		oracle.close();
	}

	@Override
	public int getMaxFieldSize() throws SQLException {
		return oracle.getMaxFieldSize();
	}

	@Override
	public void setMaxFieldSize(int max) throws SQLException {
		oracle.setMaxFieldSize(max);
	}

	@Override
	public int getMaxRows() throws SQLException {
		return oracle.getMaxRows();
	}

	@Override
	public void setMaxRows(int max) throws SQLException {
		oracle.setMaxRows(max);
	}

	@Override
	public void setEscapeProcessing(boolean enable) throws SQLException {
		oracle.setEscapeProcessing(enable);
	}

	@Override
	public int getQueryTimeout() throws SQLException {
		return oracle.getQueryTimeout();
	}

	@Override
	public void setQueryTimeout(int seconds) throws SQLException {
		oracle.setQueryTimeout(seconds);
	}

	@Override
	public void cancel() throws SQLException {
		oracle.cancel();
	}

	@Override
	public SQLWarning getWarnings() throws SQLException {
		return oracle.getWarnings();
	}

	@Override
	public void clearWarnings() throws SQLException {
		oracle.clearWarnings();
	}

	@Override
	public void setCursorName(String name) throws SQLException {
		oracle.setCursorName(name);
	}

	@Override
	public int getUpdateCount() throws SQLException {
		return oracle.getUpdateCount();
	}

	@Override
	public boolean getMoreResults() throws SQLException {
		return oracle.getMoreResults();
	}

	@Override
	public void setFetchDirection(int direction) throws SQLException {
		oracle.setFetchDirection(direction);
	}

	@Override
	public int getFetchDirection() throws SQLException {
		return oracle.getFetchDirection();
	}

	@Override
	public void setFetchSize(int rows) throws SQLException {
		oracle.setFetchSize(rows);
	}

	@Override
	public int getFetchSize() throws SQLException {
		return oracle.getFetchSize();
	}

	@Override
	public int getResultSetConcurrency() throws SQLException {
		return oracle.getResultSetConcurrency();
	}

	@Override
	public int getResultSetType() throws SQLException {
		return oracle.getResultSetType();
	}

	@Override
	public void addBatch(String sql) throws SQLException {
		oracle.addBatch(translator.translate(sql));
	}

	@Override
	public void clearBatch() throws SQLException {
		oracle.clearBatch();
	}

	@Override
	public int[] executeBatch() throws SQLException {
		return oracle.executeBatch();
	}

	@Override
	public boolean getMoreResults(int current) throws SQLException {
		return oracle.getMoreResults(current);
	}

	@Override
	public int getResultSetHoldability() throws SQLException {
		return oracle.getResultSetHoldability();
	}

	@Override
	public boolean isClosed() throws SQLException {
		return oracle.isClosed();
	}

	@Override
	public void setPoolable(boolean poolable) throws SQLException {
		oracle.setPoolable(poolable);
	}

	@Override
	public boolean isPoolable() throws SQLException {
		return oracle.isPoolable();
	}

	@Override
	public void closeOnCompletion() throws SQLException {
		oracle.closeOnCompletion();
	}

	@Override
	public boolean isCloseOnCompletion() throws SQLException {
		return oracle.isCloseOnCompletion();
	}

	@Override
	public void clearDefines() throws SQLException {
		oracle.clearDefines();
	}

	@Override
	public void closeWithKey(String key) throws SQLException {
		oracle.closeWithKey(key);
	}

	@Deprecated
	@Override
	public int creationState() {
		return oracle.creationState();
	}

	@Override
	public void defineColumnType(int columnIndex, int type) throws SQLException {
		oracle.defineColumnType(columnIndex, type);
	}

	@Override
	public void defineColumnType(int columnIndex, int type, int lobPrefetchSize) throws SQLException {
		oracle.defineColumnType(columnIndex, type, lobPrefetchSize);		
	}

	@Override
	public void defineColumnType(int columnIndex, int type, String typeName) throws SQLException {
		oracle.defineColumnType(columnIndex, type, typeName);
	}

	@Deprecated
	@Override
	public void defineColumnType(int columnIndex, int type, int lobPrefetchSize, short formOfUse) throws SQLException {
		oracle.defineColumnType(columnIndex, type, lobPrefetchSize, formOfUse);
	}

	@Deprecated
	@Override
	public void defineColumnTypeBytes(int columnIndex, int type, int lobPrefetchSize) throws SQLException {
		oracle.defineColumnTypeBytes(columnIndex, type, lobPrefetchSize);
	}

	@Deprecated
	@Override
	public void defineColumnTypeChars(int columnIndex, int type, int lobPrefetchSize) throws SQLException {
		oracle.defineColumnTypeChars(columnIndex, type, lobPrefetchSize);
	}

	@Override
	public int getLobPrefetchSize() throws SQLException {
		return oracle.getLobPrefetchSize();
	}

	@Override
	public long getRegisteredQueryId() throws SQLException {
		return oracle.getRegisteredQueryId();
	}

	@Override
	public String[] getRegisteredTableNames() throws SQLException {
		return oracle.getRegisteredTableNames();
	}

	@Override
	public int getRowPrefetch() {
		return oracle.getRowPrefetch();
	}

	@Override
	public boolean isNCHAR(int index) throws SQLException {
		return oracle.isNCHAR(index);
	}

	@Override
	public void setDatabaseChangeRegistration(DatabaseChangeRegistration registration) throws SQLException {
		oracle.setDatabaseChangeRegistration(registration);
	}

	@Override
	public void setLobPrefetchSize(int size) throws SQLException {
		oracle.setLobPrefetchSize(size);
	}

	@Override
	public void setRowPrefetch(int size) throws SQLException {
		oracle.setRowPrefetch(size);
	}

}
