/*
 * Decompiled with CFR 0.152.
 */
package org.xipki.password;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.LinkedList;
import java.util.Locale;
import java.util.Optional;
import java.util.Properties;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.xipki.password.Args;
import org.xipki.password.ConfPairs;
import org.xipki.password.OBFPasswordService;
import org.xipki.password.PBEAlgo;
import org.xipki.password.PBEPasswordService;
import org.xipki.password.PasswordCallback;
import org.xipki.password.PasswordResolver;
import org.xipki.password.PasswordResolverException;
import org.xipki.password.SecurePasswordInputPanel;

public class Passwords {
    private static final ConcurrentLinkedQueue<PasswordResolver> resolvers = new ConcurrentLinkedQueue();
    private static boolean initialized = false;
    private static PasswordResolverException initError;

    private Passwords() {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static synchronized void init() {
        if (initialized) {
            return;
        }
        try {
            Properties sysProps = System.getProperties();
            String confFile = sysProps.getProperty("XIPKI_PASSWORD_CFG");
            if (confFile == null) {
                Path p;
                String xipkiBase = sysProps.getProperty("XIPKI_BASE");
                if (xipkiBase == null) {
                    xipkiBase = "xipki";
                }
                if (Files.exists(p = Paths.get(xipkiBase, "security", "password.cfg"), new LinkOption[0])) {
                    confFile = p.toString();
                }
            }
            LinkedList<PasswordResolver> resolvers = new LinkedList<PasswordResolver>();
            resolvers.add(new OBFPasswordResolver());
            Object pbeCallback = null;
            Integer pbeIterationCount = null;
            if (confFile != null) {
                confFile = Passwords.solveVariables(confFile, 0, sysProps);
                Properties passwordCfg = new Properties();
                try (FileReader reader = new FileReader(confFile);){
                    passwordCfg.load(reader);
                    for (String propName : passwordCfg.stringPropertyNames()) {
                        PasswordResolver resolver;
                        if (!propName.startsWith("passwordResolver.")) continue;
                        String value = Passwords.solveVariables(passwordCfg.getProperty(propName).trim(), 0, sysProps);
                        int idx = value.indexOf(32);
                        String resolverClassName = idx == -1 ? value : value.substring(0, idx);
                        String resolverConf = idx == -1 ? null : value.substring(idx + 1);
                        try {
                            resolver = (PasswordResolver)Class.forName(resolverClassName).getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
                        }
                        catch (ReflectiveOperationException ex) {
                            throw new PasswordResolverException("error caught while initializing PasswordResolver " + resolverClassName + ": " + ex.getClass().getName() + ": " + ex.getMessage(), ex);
                        }
                        resolver.init(resolverConf);
                        resolvers.add(resolver);
                    }
                    String text = passwordCfg.getProperty("pbeCallback");
                    if (text != null && !text.isEmpty() && ((String)(pbeCallback = Passwords.solveVariables(text, 0, sysProps))).startsWith("FILE file=")) {
                        Path confFileParent;
                        Path filePath;
                        Object file = ((String)pbeCallback).substring("FILE file=".length());
                        if (((String)file).startsWith("~/")) {
                            file = System.getProperty("user.home") + ((String)file).substring(1);
                        }
                        if (!(filePath = Paths.get((String)file, new String[0])).isAbsolute() && (confFileParent = Paths.get(confFile, new String[0]).getParent()) != null) {
                            file = Paths.get(confFileParent.toString(), new String[]{file}).toString();
                        }
                        pbeCallback = "FILE file=" + (String)file;
                    }
                    if ((text = passwordCfg.getProperty("pbeIterationCount")) != null && !text.isEmpty()) {
                        pbeIterationCount = Integer.parseInt(Passwords.solveVariables(text, 0, sysProps));
                    }
                }
            }
            Object pbeConf = "";
            if (pbeCallback != null) {
                pbeConf = (String)pbeConf + "callback=" + ((String)pbeCallback).replace(",", "\\,").replace("=", "\\=");
            }
            if (pbeIterationCount != null) {
                if (!((String)pbeConf).isEmpty()) {
                    pbeConf = (String)pbeConf + ",";
                }
                pbeConf = (String)pbeConf + "iterationCount=" + pbeIterationCount;
            }
            PBEPasswordResolver pbe = new PBEPasswordResolver();
            pbe.init((String)pbeConf);
            resolvers.add(pbe);
            Passwords.resolvers.addAll(resolvers);
        }
        catch (PasswordResolverException ex) {
            initError = ex;
        }
        catch (Exception ex) {
            initError = new PasswordResolverException(ex);
        }
        finally {
            initialized = true;
        }
    }

    public static char[] resolvePassword(String passwordHint) throws PasswordResolverException {
        if (passwordHint == null) {
            return null;
        }
        int index = passwordHint.indexOf(58);
        if (index == -1) {
            return passwordHint.toCharArray();
        }
        Passwords.init();
        if (initError != null) {
            throw initError;
        }
        String protocol = passwordHint.substring(0, index);
        for (PasswordResolver resolver : resolvers) {
            if (!resolver.canResolveProtocol(protocol)) continue;
            return resolver.resolvePassword(passwordHint);
        }
        if ("OBF".equalsIgnoreCase(protocol) || "PBE".equalsIgnoreCase(protocol)) {
            throw new PasswordResolverException("could not find password resolver to resolve password of protocol '" + protocol + "'");
        }
        return passwordHint.toCharArray();
    }

    public static String protectPassword(String protocol, char[] password) throws PasswordResolverException {
        Passwords.init();
        if (initError != null) {
            throw initError;
        }
        for (PasswordResolver resolver : resolvers) {
            if (!resolver.canResolveProtocol(protocol)) continue;
            return resolver.protectPassword(password);
        }
        throw new PasswordResolverException("could not find password resolver to protect password of protocol '" + protocol + "'");
    }

    private static String solveVariables(String line, int offset, Properties properties) {
        int newOffset;
        String value;
        if (offset + 4 >= ((String)line).length()) {
            return line;
        }
        int startIndex = ((String)line).indexOf("${", offset);
        if (startIndex == -1) {
            return line;
        }
        int endIndex = ((String)line).indexOf("}", startIndex + 2);
        if (endIndex == -1) {
            return line;
        }
        String variable = ((String)line).substring(startIndex, endIndex + 1);
        String name = variable.substring(2, variable.length() - 1);
        boolean isEnv = false;
        if (name.startsWith("env:")) {
            isEnv = true;
            name = name.substring(4);
        } else if (name.startsWith("sys:")) {
            name = name.substring(4);
        }
        String string = value = isEnv ? System.getenv(name) : properties.getProperty(name);
        if (value != null) {
            line = ((String)line).substring(0, startIndex) + value + ((String)line).substring(endIndex + 1);
            newOffset = startIndex + value.length() + 1;
        } else {
            newOffset = endIndex + 1;
        }
        return Passwords.solveVariables((String)line, newOffset, properties);
    }

    private static PasswordCallback getPasswordCallback(String passwordCallback) throws PasswordResolverException {
        PasswordCallback pwdCallback;
        String type;
        Object conf = null;
        int delimIndex = passwordCallback.indexOf(32);
        if (delimIndex == -1) {
            type = passwordCallback;
        } else {
            type = passwordCallback.substring(0, delimIndex);
            conf = passwordCallback.substring(delimIndex + 1);
        }
        switch (type.toUpperCase(Locale.ROOT)) {
            case "FILE": {
                pwdCallback = new FilePasswordCallback();
                break;
            }
            case "GUI": {
                pwdCallback = new GuiPasswordCallback();
                break;
            }
            case "PBE-GUI": {
                pwdCallback = new PBEGuiPasswordCallback();
                break;
            }
            case "OBF": {
                pwdCallback = new OBFPasswordCallback();
                if (conf == null || Args.startsWithIgnoreCase((String)conf, "OBF:")) break;
                conf = "OBF:" + (String)conf;
                break;
            }
            default: {
                if (type.startsWith("java:")) {
                    String className = type.substring(5);
                    try {
                        pwdCallback = (PasswordCallback)Passwords.class.getClassLoader().loadClass(className).getConstructor(new Class[0]).newInstance(new Object[0]);
                        break;
                    }
                    catch (Exception e) {
                        throw new PasswordResolverException("error creating PasswordCallback of type '" + type + "'");
                    }
                }
                throw new PasswordResolverException("invalid callback type " + type);
            }
        }
        try {
            pwdCallback.init((String)conf);
        }
        catch (PasswordResolverException ex) {
            throw new IllegalArgumentException("invalid passwordCallback configuration " + passwordCallback + ", " + ex.getClass().getName() + ": " + ex.getMessage());
        }
        return pwdCallback;
    }

    private static class PBEGuiPasswordCallback
    extends GuiPasswordCallback {
        private PBEGuiPasswordCallback() {
        }

        @Override
        protected boolean isPasswordValid(char[] password, String testToken) {
            if (Args.isBlank(testToken)) {
                return true;
            }
            try {
                PBEPasswordService.decryptPassword(password, testToken);
                return true;
            }
            catch (PasswordResolverException ex) {
                return false;
            }
        }
    }

    private static class OBFPasswordCallback
    implements PasswordCallback {
        private char[] password;

        private OBFPasswordCallback() {
        }

        @Override
        public char[] getPassword(String prompt, String testToken) throws PasswordResolverException {
            return Optional.ofNullable(this.password).orElseThrow(() -> new PasswordResolverException("please initialize me first"));
        }

        @Override
        public void init(String conf) {
            Args.notBlank(conf, "conf");
            this.password = OBFPasswordService.deobfuscate(conf).toCharArray();
        }
    }

    private static class GuiPasswordCallback
    implements PasswordCallback {
        private int quorum = 1;
        private int tries = 3;

        private GuiPasswordCallback() {
        }

        protected boolean isPasswordValid(char[] password, String testToken) {
            return true;
        }

        @Override
        public char[] getPassword(String prompt, String testToken) throws PasswordResolverException {
            String tmpPrompt = prompt;
            if (Args.isBlank(tmpPrompt)) {
                tmpPrompt = "Password required";
            }
            for (int i = 0; i < this.tries; ++i) {
                char[] password;
                if (this.quorum == 1) {
                    password = Optional.ofNullable(SecurePasswordInputPanel.readPassword(tmpPrompt)).orElseThrow(() -> new PasswordResolverException("user has cancelled"));
                } else {
                    char[][] passwordParts = new char[this.quorum][];
                    for (int j = 0; j < this.quorum; ++j) {
                        String promptPart = tmpPrompt + " (part " + (j + 1) + "/" + this.quorum + ")";
                        passwordParts[j] = Optional.ofNullable(SecurePasswordInputPanel.readPassword(promptPart)).orElseThrow(() -> new PasswordResolverException("user has cancelled"));
                    }
                    password = Args.merge(passwordParts);
                }
                if (!this.isPasswordValid(password, testToken)) continue;
                return password;
            }
            throw new PasswordResolverException("Could not get the password after " + this.tries + " tries");
        }

        @Override
        public void init(String conf) throws PasswordResolverException {
            int intValue;
            if (Args.isBlank(conf)) {
                this.quorum = 1;
                return;
            }
            ConfPairs pairs = new ConfPairs(conf);
            String str = pairs.value("quorum");
            this.quorum = Integer.parseInt(str);
            if (this.quorum < 1 || this.quorum > 10) {
                throw new PasswordResolverException("quorum " + this.quorum + " is not in [1,10]");
            }
            str = pairs.value("tries");
            if (Args.isNotBlank(str) && (intValue = Integer.parseInt(str)) > 0) {
                this.tries = intValue;
            }
        }
    }

    private static class FilePasswordCallback
    implements PasswordCallback {
        private String passwordFile;

        private FilePasswordCallback() {
        }

        @Override
        public char[] getPassword(String prompt, String testToken) throws PasswordResolverException {
            if (this.passwordFile == null) {
                throw new PasswordResolverException("please initialize me first");
            }
            String passwordHint = null;
            try (BufferedReader reader = Files.newBufferedReader(Paths.get(this.passwordFile, new String[0]));){
                String line;
                while ((line = reader.readLine()) != null) {
                    if (!Args.isNotBlank(line = line.trim()) || line.startsWith("#")) continue;
                    passwordHint = line;
                    break;
                }
            }
            catch (IOException ex) {
                throw new PasswordResolverException("could not read file " + this.passwordFile, ex);
            }
            if (passwordHint == null) {
                throw new PasswordResolverException("no password is specified in file " + this.passwordFile);
            }
            return Args.startsWithIgnoreCase(passwordHint, "OBF:") ? OBFPasswordService.deobfuscate(passwordHint).toCharArray() : passwordHint.toCharArray();
        }

        @Override
        public void init(String conf) throws PasswordResolverException {
            Args.notBlank(conf, "conf");
            ConfPairs pairs = new ConfPairs(conf);
            this.passwordFile = pairs.value("file");
            if (Args.isBlank(this.passwordFile)) {
                throw new PasswordResolverException("invalid configuration " + conf + ", no file is specified");
            }
        }
    }

    private static class PBEPasswordResolver
    implements PasswordResolver {
        private char[] masterPassword;
        private final Object masterPasswordLock = new Object();
        private PasswordCallback masterPasswordCallback;
        private int iterationCount = 2000;

        @Override
        public void init(String conf) throws PasswordResolverException {
            String callback = "PBE-GUI";
            if (conf != null && !conf.isEmpty()) {
                ConfPairs pairs = new ConfPairs(conf);
                String str = pairs.value("callback");
                if (str != null && !str.isEmpty()) {
                    callback = str;
                }
                if ((str = pairs.value("iterationCount")) != null && !str.isEmpty()) {
                    this.iterationCount = Integer.parseInt(str);
                    if (this.iterationCount < 1000) {
                        throw new PasswordResolverException("iterationCount less than 1000 is not allowed");
                    }
                }
            }
            this.masterPasswordCallback = Passwords.getPasswordCallback(callback);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected char[] getMasterPassword(String encryptedPassword) throws PasswordResolverException {
            Object object = this.masterPasswordLock;
            synchronized (object) {
                if (this.masterPassword == null) {
                    if (this.masterPasswordCallback == null) {
                        throw new PasswordResolverException("masterPasswordCallback is not initialized");
                    }
                    this.masterPassword = this.masterPasswordCallback.getPassword("Please enter the master password", encryptedPassword);
                }
                return this.masterPassword;
            }
        }

        @Override
        public boolean canResolveProtocol(String protocol) {
            return "PBE".equalsIgnoreCase(protocol);
        }

        @Override
        public char[] resolvePassword(String passwordHint) throws PasswordResolverException {
            return PBEPasswordService.decryptPassword(this.getMasterPassword(passwordHint), passwordHint);
        }

        @Override
        public String protectPassword(char[] password) throws PasswordResolverException {
            return PBEPasswordService.encryptPassword(PBEAlgo.PBEWithHmacSHA256AndAES_256, this.iterationCount, this.getMasterPassword(null), password);
        }
    }

    private static class OBFPasswordResolver
    implements PasswordResolver {
        @Override
        public void init(String conf) throws PasswordResolverException {
            if (conf != null && !conf.isEmpty()) {
                throw new PasswordResolverException("non-empty conf is not allowed");
            }
        }

        @Override
        public boolean canResolveProtocol(String protocol) {
            return "OBF".equalsIgnoreCase(protocol);
        }

        @Override
        public char[] resolvePassword(String passwordHint) {
            return OBFPasswordService.deobfuscate(passwordHint).toCharArray();
        }

        @Override
        public String protectPassword(char[] password) {
            return OBFPasswordService.obfuscate(new String(password));
        }
    }
}

