/*
 * Copyright (c) 2017, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.wso2.siddhi.core.util.parser;

import org.wso2.siddhi.core.aggregation.AggregationRuntime;
import org.wso2.siddhi.core.config.SiddhiAppContext;
import org.wso2.siddhi.core.event.state.MetaStateEvent;
import org.wso2.siddhi.core.event.state.StateEventPool;
import org.wso2.siddhi.core.event.state.populater.StateEventPopulatorFactory;
import org.wso2.siddhi.core.event.stream.MetaStreamEvent;
import org.wso2.siddhi.core.event.stream.MetaStreamEvent.EventType;
import org.wso2.siddhi.core.event.stream.populater.ComplexEventPopulater;
import org.wso2.siddhi.core.event.stream.populater.StreamEventPopulaterFactory;
import org.wso2.siddhi.core.exception.StoreQueryCreationException;
import org.wso2.siddhi.core.executor.VariableExpressionExecutor;
import org.wso2.siddhi.core.query.FindStoreQueryRuntime;
import org.wso2.siddhi.core.query.SelectStoreQueryRuntime;
import org.wso2.siddhi.core.query.StoreQueryRuntime;
import org.wso2.siddhi.core.query.processor.stream.window.QueryableProcessor;
import org.wso2.siddhi.core.query.selector.QuerySelector;
import org.wso2.siddhi.core.table.Table;
import org.wso2.siddhi.core.util.SiddhiConstants;
import org.wso2.siddhi.core.util.collection.operator.CompiledCondition;
import org.wso2.siddhi.core.util.collection.operator.CompiledSelection;
import org.wso2.siddhi.core.util.collection.operator.IncrementalAggregateCompileCondition;
import org.wso2.siddhi.core.util.collection.operator.MatchingMetaInfoHolder;
import org.wso2.siddhi.core.util.parser.helper.QueryParserHelper;
import org.wso2.siddhi.core.window.Window;
import org.wso2.siddhi.query.api.aggregation.Within;
import org.wso2.siddhi.query.api.definition.AbstractDefinition;
import org.wso2.siddhi.query.api.execution.query.StoreQuery;
import org.wso2.siddhi.query.api.execution.query.input.store.AggregationInputStore;
import org.wso2.siddhi.query.api.execution.query.input.store.ConditionInputStore;
import org.wso2.siddhi.query.api.execution.query.input.store.InputStore;
import org.wso2.siddhi.query.api.execution.query.output.stream.OutputStream;
import org.wso2.siddhi.query.api.execution.query.output.stream.ReturnStream;
import org.wso2.siddhi.query.api.execution.query.selection.Selector;
import org.wso2.siddhi.query.api.expression.Expression;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Class to parse {@link StoreQueryRuntime}.
 */
public class StoreQueryParser {

    /**
     * Parse a storeQuery and return corresponding StoreQueryRuntime.
     *
     * @param storeQuery       storeQuery to be parsed.
     * @param siddhiAppContext associated Siddhi app context.
     * @param tableMap         keyvalue containing tables.
     * @param windowMap        keyvalue containing windows.
     * @param aggregationMap   keyvalue containing aggregation runtimes.
     * @return StoreQueryRuntime
     */
    public static StoreQueryRuntime parse(StoreQuery storeQuery, SiddhiAppContext siddhiAppContext,
                                          Map<String, Table> tableMap, Map<String, Window> windowMap,
                                          Map<String, AggregationRuntime> aggregationMap) {
        String queryName = "store_query_" + storeQuery.getInputStore().getStoreId();
        InputStore inputStore = storeQuery.getInputStore();
        int metaPosition = SiddhiConstants.UNKNOWN_STATE;
        Within within = null;
        Expression per = null;
        Expression onCondition = Expression.value(true);
        MetaStreamEvent metaStreamEvent = new MetaStreamEvent();
        metaStreamEvent.setInputReferenceId(inputStore.getStoreReferenceId());

        if (inputStore instanceof AggregationInputStore) {
            AggregationInputStore aggregationInputStore = (AggregationInputStore) inputStore;
            if (aggregationMap.get(inputStore.getStoreId()) == null) {
                throw new StoreQueryCreationException("Aggregation \"" + inputStore.getStoreId() + "\" has not been " +
                        "defined");
            }
            if (aggregationInputStore.getPer() != null && aggregationInputStore.getWithin() != null) {
                within = aggregationInputStore.getWithin();
                per = aggregationInputStore.getPer();
            } else if (aggregationInputStore.getPer() != null || aggregationInputStore.getWithin() != null) {
                throw new StoreQueryCreationException(
                        inputStore.getStoreId() + " should either have both 'within' and 'per' defined or none.");
            }
            if (((AggregationInputStore) inputStore).getOnCondition() != null) {
                onCondition = ((AggregationInputStore) inputStore).getOnCondition();
            }
        } else if (inputStore instanceof ConditionInputStore) {
            if (((ConditionInputStore) inputStore).getOnCondition() != null) {
                onCondition = ((ConditionInputStore) inputStore).getOnCondition();
            }
        }
        List<VariableExpressionExecutor> variableExpressionExecutors = new ArrayList<>();
        Table table = tableMap.get(inputStore.getStoreId());
        if (table != null) {
            return constructStoreQueryRuntime(table, storeQuery, siddhiAppContext, tableMap, queryName, metaPosition,
                    onCondition, metaStreamEvent, variableExpressionExecutors);
        } else {
            AggregationRuntime aggregation = aggregationMap.get(inputStore.getStoreId());
            if (aggregation != null) {
                return constructStoreQueryRuntime(aggregation, storeQuery, siddhiAppContext, tableMap, queryName,
                        within, per, onCondition, metaStreamEvent, variableExpressionExecutors);
            } else {
                Window window = windowMap.get(inputStore.getStoreId());
                if (window != null) {
                    return constructStoreQueryRuntime(window, storeQuery, siddhiAppContext, tableMap, queryName,
                            metaPosition, onCondition, metaStreamEvent, variableExpressionExecutors);
                } else {
                    throw new StoreQueryCreationException(
                            inputStore.getStoreId() + " is neither a table, aggregation or window");
                }
            }
        }
    }

    private static StoreQueryRuntime constructStoreQueryRuntime(Window window, StoreQuery storeQuery, SiddhiAppContext siddhiAppContext, Map<String, Table> tableMap, String queryName, int metaPosition, Expression onCondition, MetaStreamEvent metaStreamEvent, List<VariableExpressionExecutor> variableExpressionExecutors) {
        metaStreamEvent.setEventType(EventType.WINDOW);
        initMetaStreamEvent(metaStreamEvent, window.getWindowDefinition());
        MatchingMetaInfoHolder metaStreamInfoHolder = generateMatchingMetaInfoHolder(metaStreamEvent,
                window.getWindowDefinition());
        CompiledCondition compiledCondition = window.compileCondition(onCondition,
                generateMatchingMetaInfoHolder(metaStreamEvent, window.getWindowDefinition()),
                siddhiAppContext, variableExpressionExecutors, tableMap, queryName);
        FindStoreQueryRuntime findStoreQueryRuntime = new FindStoreQueryRuntime(window, compiledCondition,
                queryName, metaStreamEvent.getEventType());
        populateStoreQueryRuntime(findStoreQueryRuntime, metaStreamInfoHolder, storeQuery.getSelector(),
                variableExpressionExecutors, siddhiAppContext, tableMap, queryName, metaPosition);
        return findStoreQueryRuntime;
    }

    private static StoreQueryRuntime constructStoreQueryRuntime(AggregationRuntime aggregation, StoreQuery storeQuery, SiddhiAppContext siddhiAppContext, Map<String, Table> tableMap, String queryName, Within within, Expression per, Expression onCondition, MetaStreamEvent metaStreamEvent, List<VariableExpressionExecutor> variableExpressionExecutors) {
        int metaPosition;
        metaStreamEvent.setEventType(EventType.AGGREGATE);
        initMetaStreamEvent(metaStreamEvent, aggregation.getAggregationDefinition());
        MatchingMetaInfoHolder metaStreamInfoHolder = generateMatchingMetaInfoHolder(metaStreamEvent,
                aggregation.getAggregationDefinition());
        CompiledCondition compiledCondition = aggregation.compileExpression(onCondition, within, per,
                metaStreamInfoHolder, variableExpressionExecutors, tableMap, queryName, siddhiAppContext);
        metaStreamInfoHolder = aggregation.getAlteredMatchingMetaInfoHolder();
        FindStoreQueryRuntime findStoreQueryRuntime = new FindStoreQueryRuntime(aggregation, compiledCondition,
                queryName, metaStreamEvent.getEventType());
        metaPosition = 1;
        populateStoreQueryRuntime(findStoreQueryRuntime, metaStreamInfoHolder,
                storeQuery.getSelector(), variableExpressionExecutors, siddhiAppContext, tableMap,
                queryName, metaPosition);
        ComplexEventPopulater complexEventPopulater = StreamEventPopulaterFactory.constructEventPopulator(
                metaStreamInfoHolder.getMetaStateEvent().getMetaStreamEvent(0), 0,
                ((IncrementalAggregateCompileCondition) compiledCondition).getAdditionalAttributes());
        ((IncrementalAggregateCompileCondition) compiledCondition)
                .setComplexEventPopulater(complexEventPopulater);
        return findStoreQueryRuntime;
    }

    private static StoreQueryRuntime constructStoreQueryRuntime(Table table, StoreQuery storeQuery, SiddhiAppContext siddhiAppContext, Map<String, Table> tableMap, String queryName, int metaPosition, Expression onCondition, MetaStreamEvent metaStreamEvent, List<VariableExpressionExecutor> variableExpressionExecutors) {
        metaStreamEvent.setEventType(EventType.TABLE);
        initMetaStreamEvent(metaStreamEvent, table.getTableDefinition());
        MatchingMetaInfoHolder metaStreamInfoHolder = generateMatchingMetaInfoHolder(metaStreamEvent,
                table.getTableDefinition());
        CompiledCondition compiledCondition = table.compileCondition(onCondition, metaStreamInfoHolder,
                siddhiAppContext, variableExpressionExecutors, tableMap, queryName);
        if (table instanceof QueryableProcessor) {
            CompiledSelection compiledSelection = ((QueryableProcessor) table).compileSelection(
                    storeQuery.getSelector(), metaStreamInfoHolder, siddhiAppContext, variableExpressionExecutors,
                    tableMap, queryName);
            SelectStoreQueryRuntime storeQueryRuntime = new SelectStoreQueryRuntime((QueryableProcessor) table,
                    compiledCondition, compiledSelection, queryName);
            QueryParserHelper.reduceMetaComplexEvent(metaStreamInfoHolder.getMetaStateEvent());
            QueryParserHelper.updateVariablePosition(metaStreamInfoHolder.getMetaStateEvent(),
                    variableExpressionExecutors);
            return storeQueryRuntime;
        } else {
            FindStoreQueryRuntime storeQueryRuntime = new FindStoreQueryRuntime(table, compiledCondition, queryName,
                    metaStreamEvent.getEventType());
            populateStoreQueryRuntime(storeQueryRuntime, metaStreamInfoHolder, storeQuery.getSelector(),
                    variableExpressionExecutors, siddhiAppContext, tableMap, queryName, metaPosition);
            return storeQueryRuntime;
        }
    }

    private static void populateStoreQueryRuntime(FindStoreQueryRuntime findStoreQueryRuntime,
                                                  MatchingMetaInfoHolder metaStreamInfoHolder, Selector selector,
                                                  List<VariableExpressionExecutor> variableExpressionExecutors,
                                                  SiddhiAppContext siddhiAppContext,
                                                  Map<String, Table> tableMap, String queryName, int metaPosition) {
        QuerySelector querySelector = SelectorParser.parse(selector,
                new ReturnStream(OutputStream.OutputEventType.CURRENT_EVENTS), siddhiAppContext,
                metaStreamInfoHolder.getMetaStateEvent(), tableMap, variableExpressionExecutors, queryName,
                metaPosition);
        QueryParserHelper.reduceMetaComplexEvent(metaStreamInfoHolder.getMetaStateEvent());
        QueryParserHelper.updateVariablePosition(metaStreamInfoHolder.getMetaStateEvent(), variableExpressionExecutors);
        querySelector.setEventPopulator(
                StateEventPopulatorFactory.constructEventPopulator(metaStreamInfoHolder.getMetaStateEvent()));
        findStoreQueryRuntime.setStateEventPool(new StateEventPool(metaStreamInfoHolder.getMetaStateEvent(), 5));
        findStoreQueryRuntime.setSelector(querySelector);
    }

    private static MatchingMetaInfoHolder generateMatchingMetaInfoHolder(MetaStreamEvent metaStreamEvent,
                                                                         AbstractDefinition definition) {
        MetaStateEvent metaStateEvent = new MetaStateEvent(1);
        metaStateEvent.addEvent(metaStreamEvent);
        return new MatchingMetaInfoHolder(metaStateEvent, -1, 0, definition,
                definition, 0);
    }

    private static void initMetaStreamEvent(MetaStreamEvent metaStreamEvent, AbstractDefinition inputDefinition) {
        metaStreamEvent.addInputDefinition(inputDefinition);
        metaStreamEvent.initializeAfterWindowData();
        inputDefinition.getAttributeList().forEach(metaStreamEvent::addData);
    }

}
