/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.wicket.markup.resolver;

import org.apache.wicket.Component;
import org.apache.wicket.MarkupContainer;
import org.apache.wicket.markup.ComponentTag;
import org.apache.wicket.markup.MarkupStream;
import org.apache.wicket.markup.WicketTag;
import org.apache.wicket.markup.html.TransparentWebMarkupContainer;
import org.apache.wicket.markup.parser.filter.WicketTagIdentifier;

/**
 * Detect &lt;wicket:extend&gt; and &lt;wicket:child&gt; tags, which are silently ignored, because
 * they have already been processed.
 * 
 * @author Juergen Donnerstag
 */
public class MarkupInheritanceResolver implements IComponentResolver
{
	private static final long serialVersionUID = 1L;

	/** */
	public static final String CHILD = "child";

	/** */
	public static final String EXTEND = "extend";

	static
	{
		// register "wicket:extend" and "wicket:child"
		WicketTagIdentifier.registerWellKnownTagName(EXTEND);
		WicketTagIdentifier.registerWellKnownTagName(CHILD);
	}

	/**
	 * @see org.apache.wicket.markup.resolver.IComponentResolver#resolve(org.apache.wicket.MarkupContainer,
	 *      org.apache.wicket.markup.MarkupStream, org.apache.wicket.markup.ComponentTag)
	 */
	@Override
	public Component resolve(final MarkupContainer container, final MarkupStream markupStream,
		final ComponentTag tag)
	{
		// It must be <wicket:...>
		if (tag instanceof WicketTag)
		{
			final WicketTag wicketTag = (WicketTag)tag;

			// It must be <wicket:extend...>
			if (wicketTag.isExtendTag() || wicketTag.isChildTag())
			{
				String id = wicketTag.getId() + container.getPage().getAutoIndex();
				return new TransparentWebMarkupContainer(id);
			}
		}

		// We were not able to handle the componentId
		return null;
	}
}