/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.windowing;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Logger;
import org.apache.storm.windowing.Event;
import org.apache.storm.windowing.EventImpl;
import org.apache.storm.windowing.EvictionPolicy;
import org.apache.storm.windowing.TriggerHandler;
import org.apache.storm.windowing.TriggerPolicy;
import org.apache.storm.windowing.WindowLifecycleListener;

public class WindowManager<T>
implements TriggerHandler {
    public static final int EXPIRE_EVENTS_THRESHOLD = 100;
    private static final Logger LOG = Logger.getLogger(WindowManager.class.getName());
    private static final String EVICTION_STATE_KEY = "es";
    private static final String TRIGGER_STATE_KEY = "ts";
    protected final Collection<Event<T>> queue;
    protected final WindowLifecycleListener<T> windowLifecycleListener;
    private final List<T> expiredEvents;
    private final Set<Event<T>> prevWindowEvents;
    private final AtomicInteger eventsSinceLastExpiry;
    private final ReentrantLock lock;
    protected EvictionPolicy<T, ?> evictionPolicy;
    protected TriggerPolicy<T, ?> triggerPolicy;

    public WindowManager(WindowLifecycleListener<T> lifecycleListener) {
        this(lifecycleListener, new ConcurrentLinkedQueue<Event<T>>());
    }

    public WindowManager(WindowLifecycleListener<T> lifecycleListener, Collection<Event<T>> queue) {
        this.windowLifecycleListener = lifecycleListener;
        this.queue = queue;
        this.expiredEvents = new ArrayList<T>();
        this.prevWindowEvents = new HashSet<Event<T>>();
        this.eventsSinceLastExpiry = new AtomicInteger();
        this.lock = new ReentrantLock(true);
    }

    public void setEvictionPolicy(EvictionPolicy<T, ?> evictionPolicy) {
        this.evictionPolicy = evictionPolicy;
    }

    public void setTriggerPolicy(TriggerPolicy<T, ?> triggerPolicy) {
        this.triggerPolicy = triggerPolicy;
    }

    public void add(T event) {
        this.add(event, System.currentTimeMillis());
    }

    public void add(T event, long ts) {
        this.add(new EventImpl<T>(event, ts));
    }

    public void add(Event<T> windowEvent) {
        if (!windowEvent.isWatermark()) {
            this.queue.add(windowEvent);
        } else {
            LOG.fine(() -> String.format("Got watermark event with ts %s", windowEvent.getTimestamp()));
        }
        this.track(windowEvent);
        this.compactWindow();
    }

    @Override
    public boolean onTrigger() {
        List<Event<T>> windowEvents = null;
        ArrayList<T> expired = null;
        try {
            this.lock.lock();
            windowEvents = this.scanEvents(true);
            expired = new ArrayList<T>(this.expiredEvents);
            this.expiredEvents.clear();
        }
        finally {
            this.lock.unlock();
        }
        ArrayList events = new ArrayList();
        ArrayList<T> newEvents = new ArrayList<T>();
        for (Event<T> event : windowEvents) {
            events.add(event.get());
            if (this.prevWindowEvents.contains(event)) continue;
            newEvents.add(event.get());
        }
        this.prevWindowEvents.clear();
        if (!events.isEmpty()) {
            this.prevWindowEvents.addAll(windowEvents);
            LOG.fine(() -> String.format("invoking windowLifecycleListener onActivation, [%s] events in window.", events.size()));
            this.windowLifecycleListener.onActivation(events, newEvents, expired, this.evictionPolicy.getContext().getReferenceTime());
        } else {
            LOG.fine("No events in the window, skipping onActivation");
        }
        this.triggerPolicy.reset();
        return !events.isEmpty();
    }

    public void shutdown() {
        LOG.fine("Shutting down WindowManager");
        if (this.triggerPolicy != null) {
            this.triggerPolicy.shutdown();
        }
    }

    protected void compactWindow() {
        if (this.eventsSinceLastExpiry.incrementAndGet() >= 100) {
            this.scanEvents(false);
        }
    }

    private void track(Event<T> windowEvent) {
        this.evictionPolicy.track(windowEvent);
        this.triggerPolicy.track(windowEvent);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<Event<T>> scanEvents(boolean fullScan) {
        LOG.fine(() -> String.format("Scan events, eviction policy %s", this.evictionPolicy));
        ArrayList eventsToExpire = new ArrayList();
        ArrayList<Event<T>> eventsToProcess = new ArrayList<Event<T>>();
        try {
            this.lock.lock();
            Iterator<Event<T>> it = this.queue.iterator();
            while (it.hasNext()) {
                Event<T> windowEvent = it.next();
                EvictionPolicy.Action action = this.evictionPolicy.evict(windowEvent);
                if (action == EvictionPolicy.Action.EXPIRE) {
                    eventsToExpire.add(windowEvent.get());
                    it.remove();
                    continue;
                }
                if (!fullScan || action == EvictionPolicy.Action.STOP) break;
                if (action != EvictionPolicy.Action.PROCESS) continue;
                eventsToProcess.add(windowEvent);
            }
            this.expiredEvents.addAll(eventsToExpire);
        }
        finally {
            this.lock.unlock();
        }
        this.eventsSinceLastExpiry.set(0);
        LOG.fine(() -> String.format("[%d] events expired from window.", eventsToExpire.size()));
        if (!eventsToExpire.isEmpty()) {
            LOG.fine("invoking windowLifecycleListener.onExpiry");
            this.windowLifecycleListener.onExpiry(eventsToExpire);
        }
        return eventsToProcess;
    }

    public long getEarliestEventTs(long startTs, long endTs) {
        long minTs = Long.MAX_VALUE;
        for (Event<T> event : this.queue) {
            if (event.getTimestamp() <= startTs || event.getTimestamp() > endTs) continue;
            minTs = Math.min(minTs, event.getTimestamp());
        }
        return minTs;
    }

    public int getEventCount(long referenceTime) {
        int count = 0;
        for (Event<T> event : this.queue) {
            if (event.getTimestamp() > referenceTime) continue;
            ++count;
        }
        return count;
    }

    public List<Long> getSlidingCountTimestamps(long startTs, long endTs, int slidingCount) {
        ArrayList<Long> timestamps = new ArrayList<Long>();
        if (endTs > startTs) {
            int count = 0;
            long ts = Long.MIN_VALUE;
            for (Event<T> event : this.queue) {
                if (event.getTimestamp() <= startTs || event.getTimestamp() > endTs) continue;
                ts = Math.max(ts, event.getTimestamp());
                if (++count % slidingCount != 0) continue;
                timestamps.add(ts);
            }
        }
        return timestamps;
    }

    public String toString() {
        return "WindowManager{evictionPolicy=" + this.evictionPolicy + ", triggerPolicy=" + this.triggerPolicy + '}';
    }

    public void restoreState(Map<String, Optional<?>> state) {
        Optional.ofNullable(state.get(EVICTION_STATE_KEY)).flatMap(x -> x).ifPresent(v -> this.evictionPolicy.restoreState(v));
        Optional.ofNullable(state.get(TRIGGER_STATE_KEY)).flatMap(x -> x).ifPresent(v -> this.triggerPolicy.restoreState(v));
    }

    public Map<String, Optional<?>> getState() {
        HashMap stateMap = new HashMap();
        stateMap.put(EVICTION_STATE_KEY, Optional.ofNullable(this.evictionPolicy.getState()));
        stateMap.put(TRIGGER_STATE_KEY, Optional.ofNullable(this.triggerPolicy.getState()));
        return Collections.unmodifiableMap(stateMap);
    }
}

