/**
 * The MIT License (MIT)
 *
 * Copyright (c) 2020 tools4j.org (Marco Terzer, Anton Anufriev)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.tools4j.elara.plugin.base;

import org.tools4j.elara.application.CommandProcessor;
import org.tools4j.elara.application.EventApplier;
import org.tools4j.elara.input.Input;
import org.tools4j.elara.output.Output;
import org.tools4j.elara.plugin.api.Plugin;
import org.tools4j.elara.plugin.api.SystemPlugin;
import org.tools4j.elara.plugin.api.TypeRange;
import org.tools4j.elara.plugin.base.BaseState.Mutable;

import static java.util.Objects.requireNonNull;

/**
 * Default plugin to initialise {@link BaseState}.  Another plugin can be used to initialise the base state if it
 * returns an implementation of {@link BaseContext}.
 */
public enum BasePlugin implements SystemPlugin<Mutable> {
    INSTANCE;

    @Override
    public Mutable defaultPluginState() {
        return BaseContext.createDefaultBaseState();
    }

    @Override
    public BaseContext configuration(final org.tools4j.elara.init.Configuration appConfig, final Mutable baseState) {
        requireNonNull(appConfig);
        requireNonNull(baseState);
        return () -> baseState;
    }

    @Override
    public TypeRange typeRange() {
        return TypeRange.BASE;
    }

    /**
     * Base context to initialise base state.  Other plugins can implement this
     * context if they want to replace the default base plugin and extend the base
     * state.
     */
    @FunctionalInterface
    public interface BaseContext extends Configuration {
        static BaseState.Mutable createDefaultBaseState() {
            return new DefaultBaseState();
        }

        BaseState.Mutable baseState();

        @Override
        default Input[] inputs(final BaseState baseState) {
            return NO_INPUTS;
        }

        @Override
        default Output output(final BaseState baseState) {
            return Output.NOOP;
        }

        @Override
        default CommandProcessor commandProcessor(final BaseState baseState) {
            return CommandProcessor.NOOP;
        }

        @Override
        default EventApplier eventApplier(final BaseState.Mutable baseState) {
            return EventApplier.NOOP;
        }
    }
}
