package org.ton.cell

import org.ton.bigint.*
import org.ton.bitstring.BitString
import org.ton.bitstring.ByteBackedMutableBitString
import org.ton.bitstring.MutableBitString
import org.ton.cell.exception.CellOverflowException
import kotlin.contracts.ExperimentalContracts
import kotlin.contracts.InvocationKind
import kotlin.contracts.contract
import kotlin.jvm.JvmStatic

public interface CellBuilder {
    public var bits: MutableBitString
    public var refs: MutableList<Cell>
    public var isExotic: Boolean

    public val bitsPosition: Int

    /**
     * Converts a builder into an ordinary cell.
     */
    public fun endCell(): Cell

    public fun storeBit(bit: Boolean): CellBuilder
    public fun storeBits(vararg bits: Boolean): CellBuilder
    public fun storeBits(bits: Iterable<Boolean>): CellBuilder
    public fun storeBits(bits: Collection<Boolean>): CellBuilder
    public fun storeBits(bits: BitString): CellBuilder

    public fun storeBytes(byteArray: ByteArray): CellBuilder

    /**
     * Stores a reference to cell into builder.
     */
    public fun storeRef(ref: Cell): CellBuilder

    public fun storeRefs(vararg refs: Cell): CellBuilder
    public fun storeRefs(refs: Iterable<Cell>): CellBuilder
    public fun storeRefs(refs: Collection<Cell>): CellBuilder

    /**
     * Stores an unsigned [length]-bit integer [value] into builder for 0 ≤ [length] ≤ 256.
     */
    public fun storeUInt(value: BigInt, length: Int): CellBuilder
    public fun storeUInt(value: Byte, length: Int): CellBuilder = storeUInt(BigInt(value), length)
    public fun storeUInt(value: Short, length: Int): CellBuilder = storeUInt(BigInt(value), length)
    public fun storeUInt(value: Int, length: Int): CellBuilder = storeUInt(BigInt(value), length)
    public fun storeUInt(value: Long, length: Int): CellBuilder = storeUInt(BigInt(value), length)

    public fun storeUInt8(value: UByte): CellBuilder = storeInt(value.toByte(), 8)
    public fun storeUInt16(value: UShort): CellBuilder = storeInt(value.toShort(), 16)
    public fun storeUInt32(value: UInt): CellBuilder = storeInt(value.toInt(), 32)
    public fun storeUInt64(value: ULong): CellBuilder = storeInt(value.toLong(), 64)

    public fun storeUIntLeq(value: BigInt, max: BigInt): CellBuilder = storeUInt(value, max.bitLength)
    public fun storeUIntLeq(value: Byte, max: Byte): CellBuilder = storeUIntLeq(BigInt(value), BigInt(max))
    public fun storeUIntLeq(value: Short, max: Short): CellBuilder = storeUIntLeq(BigInt(value), BigInt(max))
    public fun storeUIntLeq(value: Int, max: Int): CellBuilder = storeUIntLeq(BigInt(value), BigInt(max))
    public fun storeUIntLeq(value: Long, max: Long): CellBuilder = storeUIntLeq(BigInt(value), BigInt(max))

    public fun storeUIntLes(value: BigInt, max: BigInt): CellBuilder = storeUInt(value, (max - 1).bitLength)
    public fun storeUIntLes(value: Byte, max: Byte): CellBuilder = storeUIntLes(BigInt(value), BigInt(max))
    public fun storeUIntLes(value: Short, max: Short): CellBuilder = storeUIntLes(BigInt(value), BigInt(max))
    public fun storeUIntLes(value: Int, max: Int): CellBuilder = storeUIntLes(BigInt(value), BigInt(max))
    public fun storeUIntLes(value: Long, max: Long): CellBuilder = storeUIntLes(BigInt(value), BigInt(max))

    /**
     * Stores a signed [length]-bit integer [value] into builder for 0 ≤ [length] ≤ 257.
     */
    public fun storeInt(value: BigInt, length: Int): CellBuilder
    public fun storeInt(value: Byte, length: Int): CellBuilder = storeInt(BigInt(value), length)
    public fun storeInt(value: Short, length: Int): CellBuilder = storeInt(BigInt(value), length)
    public fun storeInt(value: Int, length: Int): CellBuilder = storeInt(BigInt(value), length)
    public fun storeInt(value: Long, length: Int): CellBuilder = storeInt(BigInt(value), length)

    /**
     * Stores [slice] into builder.
     */
    public fun storeSlice(slice: CellSlice): CellBuilder

    public companion object {
        @JvmStatic
        public fun of(cell: Cell): CellBuilder =
            CellBuilderImpl(cell.bits.toMutableBitString(), cell.refs.toMutableList())

        @JvmStatic
        public fun beginCell(): CellBuilder = CellBuilderImpl()

        @JvmStatic
        public fun createCell(builder: CellBuilder.() -> Unit): Cell =
            CellBuilderImpl().apply(builder).endCell()

        @JvmStatic
        public fun createPrunedBranch(cell: Cell, newLevel: Int, virtualizationLevel: Int = Cell.MAX_LEVEL): Cell =
            createCell {
                val levelMask = cell.levelMask.apply(virtualizationLevel)
                val level = levelMask.level
                check(newLevel >= level + 1)

                storeUInt(CellType.PRUNED_BRANCH.value, 8)
                storeUInt((levelMask or LevelMask.level(newLevel)).mask, 8)
                repeat(level + 1) {
                    if (levelMask.isSignificant(it)) {
                        storeBytes(cell.hash(it))
                    }
                }
                repeat(level + 1) {
                    if (levelMask.isSignificant(it)) {
                        storeUInt(cell.depth(it), 16)
                    }
                }
            }

        @JvmStatic
        public fun createMerkleProof(cellProof: Cell): Cell = createCell {
            storeUInt(CellType.MERKLE_PROOF.value, 8)
            storeBytes(cellProof.hash(level = 0))
            storeUInt(cellProof.depth(level = 0), Cell.DEPTH_BITS)
            storeRef(cellProof)
        }

        @JvmStatic
        public fun createMerkleUpdate(fromProof: Cell, toProof: Cell): Cell = createCell {
            storeUInt(CellType.MERKLE_UPDATE.value, 8)
            storeBytes(fromProof.hash(level = 0))
            storeBytes(toProof.hash(level = 0))
            storeUInt(fromProof.depth(level = 0), Cell.DEPTH_BITS)
            storeUInt(toProof.depth(level = 0), Cell.DEPTH_BITS)
            storeRef(fromProof)
            storeRef(toProof)
        }
    }

    public fun storeBytes(byteArray: ByteArray, length: Int): CellBuilder
}

public inline operator fun CellBuilder.invoke(builder: CellBuilder.() -> Unit) {
    builder(this)
}

public inline fun CellBuilder.storeRef(refBuilder: CellBuilder.() -> Unit): CellBuilder = apply {
    val cellBuilder = CellBuilder.beginCell()
    cellBuilder.apply(refBuilder)
    val cell = cellBuilder.endCell()
    storeRef(cell)
}

public fun CellBuilder(cell: Cell): CellBuilder =
    CellBuilder.of(cell)

@OptIn(ExperimentalContracts::class)
public fun CellBuilder(builder: CellBuilder.() -> Unit = {}): CellBuilder {
    contract {
        callsInPlace(builder, InvocationKind.EXACTLY_ONCE)
    }
    return CellBuilderImpl().apply(builder)
}

private class CellBuilderImpl(
    override var bits: MutableBitString = ByteBackedMutableBitString.of(),
    override var refs: MutableList<Cell> = ArrayList()
) : CellBuilder {
    private val remainder: Int get() = Cell.MAX_BITS_SIZE - bitsPosition
    override val bitsPosition: Int get() = bits.size
    override var isExotic: Boolean = false

    override fun endCell(): Cell = Cell(bits, refs, isExotic)

    override fun storeBit(bit: Boolean): CellBuilder = apply {
        checkBitsOverflow(1)
        bits.plus(bit)
    }

    override fun storeBits(vararg bits: Boolean): CellBuilder = apply {
        checkBitsOverflow(bits.size)
        this.bits += bits
    }

    override fun storeBits(bits: Collection<Boolean>): CellBuilder = apply {
        checkBitsOverflow(bits.size)
        this.bits.plus(bits)
    }

    override fun storeBits(bits: BitString): CellBuilder = apply {
        checkBitsOverflow(bits.size)
        this.bits.plus(bits)
    }

    override fun storeBits(bits: Iterable<Boolean>): CellBuilder = storeBits(bits.toList())

    override fun storeBytes(byteArray: ByteArray): CellBuilder = apply {
        checkBitsOverflow(byteArray.size * Byte.SIZE_BITS)
        this.bits.plus(byteArray)
    }

    override fun storeBytes(byteArray: ByteArray, length: Int): CellBuilder = apply {
        checkBitsOverflow(length)
        this.bits.plus(byteArray, length)
    }

    override fun storeRef(ref: Cell): CellBuilder = apply {
        checkRefsOverflow(1)
        refs.add(ref)
    }

    override fun storeRefs(vararg refs: Cell): CellBuilder = apply {
        checkRefsOverflow(refs.size)
        this.refs.addAll(refs.filter { !it.bits.isEmpty() })
    }

    override fun storeRefs(refs: Iterable<Cell>): CellBuilder = storeRefs(refs.toList())

    override fun storeRefs(refs: Collection<Cell>): CellBuilder = apply {
        checkRefsOverflow(refs.size)
        this.refs.addAll(refs.filter { !it.bits.isEmpty() })
    }

    override fun storeUInt(value: BigInt, length: Int): CellBuilder = apply {
        check(value.bitLength <= length) { "Integer `$value` does not fit into $length bits" }
        check(value.sign >= 0) { "Integer `$value` must be unsigned" }
        storeNumber(value, length)
    }

    override fun storeInt(value: BigInt, length: Int): CellBuilder = apply {
        val intBits = BigInt(1) shl (length - 1)
        require(value >= -intBits && value < intBits) { "Can't store an Int, because its value allocates more space than provided." }
        storeNumber(value, length)
    }

    private fun storeNumber(value: BigInt, length: Int): CellBuilder = apply {
        val bits = BooleanArray(length) { index ->
            ((value shr index) and BigInt(1)).toInt() == 1
        }.reversedArray()
        storeBits(*bits)
    }

    override fun storeSlice(slice: CellSlice): CellBuilder = apply {
        val (bits, refs) = slice

        checkBitsOverflow(bits.size)
        checkRefsOverflow(refs.size)

        storeBits(bits)
        refs.forEach { ref ->
            storeRef(ref)
        }
    }

    override fun toString(): String = endCell().toString()

    private fun checkBitsOverflow(length: Int) = require(length <= remainder) {
        throw CellOverflowException("Bits overflow. Can't add $length bits. $remainder bits left. - ${bits.size}")
    }

    private fun checkRefsOverflow(count: Int) = require(count <= (4 - refs.size)) {
        throw CellOverflowException("Refs overflow. Can't add $count refs. ${4 - refs.size} refs left.")
    }
}
