/*
 * Tentackle - http://www.tentackle.org.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.tentackle.sql.metadata;

import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Objects;
import org.tentackle.common.BasicStringHelper;
import org.tentackle.sql.Backend;
import org.tentackle.sql.SqlType;


/**
 * Meta data for a column
 *
 * @author harald
 */
public class ColumnMetaData implements Cloneable {

  protected final TableMetaData tableMetaData;    // the table this column belongs to
  protected String columnName;                    // the column columnName
  protected String comment;                       // the comment
  protected int type;                             // the type code from java.sql.Types
  protected String typeName;                      // backend-specific type columnName
  protected int size;                             // the size / length (for ex. in VARCHAR)
  protected int scale;                            // the scale (for ex. DECIMAL)
  protected String defaultValue;                  // column default, null = none
  protected boolean nullable;                     // true if column may contain nulls
  protected boolean generated;                    // true if column is generated by the backend
  protected boolean autoIncrement;                // true if column is autoincremented by the backend


  /**
   * Creates column meta data.
   *
   * @param tableMetaData the table meta data this column belongs to
   */
  public ColumnMetaData(TableMetaData tableMetaData) {
    this.tableMetaData = tableMetaData;
  }


  /**
   * Gets the table meta data.
   *
   * @return the table this column belongs to
   */
  public TableMetaData getTableMetaData() {
    return tableMetaData;
  }


  /**
   * Sets up the column from the database meta data result.
   *
   * @param resultSet the column result set
   * @throws SQLException the processing the result set failed
   */
  public void setupColumnFromMetaData(ResultSet resultSet) throws SQLException {
    columnName = BasicStringHelper.toLower(resultSet.getString("COLUMN_NAME"));
    comment = resultSet.getString("REMARKS");
    size = resultSet.getInt("COLUMN_SIZE");
    scale = resultSet.getShort("DECIMAL_DIGITS");
    nullable = resultSet.getInt("NULLABLE") == DatabaseMetaData.columnNullable;
    type = resultSet.getInt("DATA_TYPE");
    typeName = resultSet.getString("TYPE_NAME");
    defaultValue = resultSet.getString("COLUMN_DEF");
    autoIncrement = Objects.equals("YES", resultSet.getString("IS_AUTOINCREMENT"));
    try {
      generated = Objects.equals("YES", resultSet.getString("IS_GENERATEDCOLUMN"));
    }
    catch (SQLException sqx) {
      // backend does not support this, since JDBC 4.1 (Java7)
    }
    validate();
  }


  /**
   * Gets the columnName of the column.
   *
   * @return the columnName (always in lowercase)
   */
  public String getColumnName() {
    return columnName;
  }

  /**
   * Gets the comment.
   *
   * @return the comment, null if none
   */
  public String getComment() {
    return comment;
  }

  /**
   * Gets the column's type.
   *
   * @return the type
   * @see java.sql.Types
   */
  public int getType() {
    return type;
  }

  /**
   * Gets the backend-specific type columnName.
   *
   * @return the type columnName
   */
  public String getTypeName() {
    return typeName;
  }

  /**
   * Gets the column's size.
   *
   * @return the size
   */
  public int getSize() {
    return size;
  }

  /**
   * Gets the numeric scale.
   *
   * @return the scale
   */
  public int getScale() {
    return scale;
  }

  /**
   * Gets the column's default.
   *
   * @return the defaultValue
   */
  public String getDefaultValue() {
    return defaultValue;
  }

  /**
   * Returns whether column is nullable.
   *
   * @return true if nullable
   */
  public boolean isNullable() {
    return nullable;
  }

  /**
   * Returns whether column is generated by the backend.
   *
   * @return true if generated
   */
  public boolean isGenerated() {
    return generated;
  }

  /**
   * Returns whether column is auto-incremented by the backend.
   *
   * @return true if autoincremented
   */
  public boolean isAutoIncrement() {
    return autoIncrement;
  }


  /**
   * Checks whether colunm's type matches given SqlType according to the backend.
   *
   * @param sqlType the sqltype
   * @return true if matches
   */
  public boolean matchesSqlType(SqlType sqlType) {
    Backend backend = tableMetaData.getModelMetaData().getBackend();
    SqlType[] possibleSqlTypes = backend.jdbcTypeToSqlType(type, size, scale);
    if (possibleSqlTypes != null) {
      for (SqlType possibleType: possibleSqlTypes) {
        if (sqlType == possibleType) {
          return true;
        }
      }
    }
    return false;
  }


  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(columnName);
    buf.append(" ");
    buf.append(MetaDataHelper.jdbcTypeToString(type));
    if (size > 0) {
      buf.append("(");
      buf.append(size);
      if (scale > 0) {
        buf.append(",");
        buf.append(scale);
      }
      buf.append(")");
    }
    if (!nullable) {
      buf.append(" NOT NULL");
    }
    if (defaultValue != null) {
      buf.append(" DEFAULT ");
      buf.append(defaultValue);
    }
    if (autoIncrement) {
      buf.append(" AUTOINCREMENT");
    }
    buf.append(" -- ");
    buf.append(typeName);
    if (generated) {
      buf.append(", GENERATED");
    }
    return buf.toString();
  }


  /**
   * Validates and postprocesses the column data.
   */
  public void validate() {
    if (comment != null && comment.isEmpty()) {
      comment = null;
    }
    if (defaultValue != null && defaultValue.isEmpty()) {
      defaultValue = null;
    }
  }

}
