/*
 * Tentackle - https://tentackle.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.tentackle.persist.lock;

import org.tentackle.common.Freezable;
import org.tentackle.common.Timestamp;
import org.tentackle.dbms.AbstractDbObject;
import org.tentackle.dbms.Db;
import org.tentackle.dbms.DbObjectClassVariables;
import org.tentackle.dbms.PreparedStatementWrapper;
import org.tentackle.dbms.ResultSetWrapper;
import org.tentackle.persist.TokenLock;
import org.tentackle.session.ClassId;
import org.tentackle.session.Persistent;
import org.tentackle.session.TableName;
import org.tentackle.sql.Backend;

import java.io.Serial;

/*
 * @> $mapfile
 *
 * # token lock table
 * name := $classname
 * id := $classid
 * table := $tablename
 *
 * ## attributes
 * [NOPKEY]
 * # ID corresponds to the PDO's id
 * int            pdoClassId        pdoclassid        class ID of the PDO
 * # copy of the token from the PDO (names edited... blocked by model api)
 * long           lockedBy          lockedby          userId of token lock holder
 * Timestamp      lockedSince       lockedsince       time since token lock given to user
 * Timestamp      lockExpiry        lockexpiry        time when token lock expires
 *
 * ## indexes
 * unique index pdo := id, pdoclassid
 *
 * @<
 */


/**
 * Table holding the token locks.
 * <p>
 * A TokenLock is an add-on to the editedBy/Since/Expiry-token in PDOs.<br>
 * TokenLocks are maintained by the LockManager.<br>
 * The ID of the TokenLock is the same as the PDO being locked.<br>
 * The serial is not related to the PDO, starts at 1 and is incremented whenever lockedBy,
 * lockedSince or lockExpiry changes.<br>
 * Notice the [NOPKEY] to suppress the generation of the default primary key on id.
 * Instead, the index is unique on (id, pdoclassid). Thus, it is still allowed that
 * objects of different classes share the same id.
 * <p>
 * Notice, that this table is only used in servers, never in remote clients or simple 2-tier applications.
 *
 * @author harald
 */
@ClassId(/**/10/**/)                // @wurblet < Inject $classid
@TableName(/**/"tokenlock"/**/)     // @wurblet < Inject --string $tablename
public class DbTokenLock extends AbstractDbObject<DbTokenLock> implements TokenLock {

  @Serial
  private static final long serialVersionUID = 1L;

  /** Variables common to all instances of {@link DbTokenLock}. */
  public static final DbObjectClassVariables<DbTokenLock> CLASSVARIABLES =
      DbObjectClassVariables.create(DbTokenLock.class);


  // @wurblet fieldnames ColumnNames --model=$mapfile

  //<editor-fold defaultstate="collapsed" desc="code 'fieldnames' generated by wurblet ColumnNames">//GEN-BEGIN:fieldnames


  /** database column name for 'pdoClassId'. */
  public static final String CN_PDOCLASSID = "pdoclassid";

  /** database column name for 'lockedBy'. */
  public static final String CN_LOCKEDBY = "lockedby";

  /** database column name for 'lockedSince'. */
  public static final String CN_LOCKEDSINCE = "lockedsince";

  /** database column name for 'lockExpiry'. */
  public static final String CN_LOCKEXPIRY = "lockexpiry";

  //</editor-fold>//GEN-END:fieldnames

  // @wurblet fieldlengths ColumnLengths --model=$mapfile

//<editor-fold defaultstate="collapsed" desc="code 'fieldlengths' generated by wurblet ColumnLengths">//GEN-BEGIN:fieldlengths


//</editor-fold>//GEN-END:fieldlengths

  // @wurblet declare Declare --model=$mapfile

  //<editor-fold defaultstate="collapsed" desc="code 'declare' generated by wurblet Declare">//GEN-BEGIN:declare


  /** class ID of the PDO. */
  private int pdoClassId;

  /** userId of token lock holder. */
  private long lockedBy;

  /** time since token lock given to user. */
  private Timestamp lockedSince;

  /** time when token lock expires. */
  private Timestamp lockExpiry;

  //</editor-fold>//GEN-END:declare



  /**
   * Creates an empty modification object.
   *
   * @param db the session
   */
  public DbTokenLock(Db db) {
    super(db);
  }


  /**
   * Creates an empty modification object.
   */
  public DbTokenLock() {
    super();
  }


  @Override
  public DbObjectClassVariables<DbTokenLock> getClassVariables() {
    return CLASSVARIABLES;
  }



  // @wurblet methods MethodsImpl --noif --model=$mapfile

  //<editor-fold defaultstate="collapsed" desc="code 'methods' generated by wurblet MethodsImpl">//GEN-BEGIN:methods


  @Override
  public void getFields(ResultSetWrapper rs) {
    super.getFields(rs);
    if (rs.configureSection(CLASSVARIABLES)) {
      rs.configureColumn(CN_PDOCLASSID);
      rs.configureColumn(CN_LOCKEDBY);
      rs.configureColumn(CN_LOCKEDSINCE);
      rs.configureColumn(CN_LOCKEXPIRY);
      rs.configureColumn(CN_ID);
      rs.configureColumn(CN_SERIAL);
    }
    pdoClassId = rs.getInt();
    lockedBy = rs.getLong();
    lockedSince = rs.getTimestamp();
    lockExpiry = rs.getTimestamp();
    setId(rs.getLong());
    setSerial(rs.getLong());
  }

  @Override
  public int setFields(PreparedStatementWrapper st) {
    int ndx = super.setFields(st);
    st.setInt(++ndx, pdoClassId);
    st.setLong(++ndx, lockedBy);
    st.setTimestamp(++ndx, lockedSince);
    st.setTimestamp(++ndx, lockExpiry);
    st.setLong(++ndx, getId());
    st.setLong(++ndx, getSerial());
    return ndx;
  }

  @Override
  public String createInsertSql(Backend backend) {
    return Backend.SQL_INSERT_INTO + getTableName() + Backend.SQL_LEFT_PARENTHESIS +
           CN_PDOCLASSID + Backend.SQL_COMMA +
           CN_LOCKEDBY + Backend.SQL_COMMA +
           CN_LOCKEDSINCE + Backend.SQL_COMMA +
           CN_LOCKEXPIRY + Backend.SQL_COMMA +
           CN_ID + Backend.SQL_COMMA +
           CN_SERIAL +
           Backend.SQL_INSERT_VALUES +
           Backend.SQL_PAR_COMMA.repeat(5) +
           Backend.SQL_PAR + Backend.SQL_RIGHT_PARENTHESIS;
  }

  @Override
  public String createUpdateSql(Backend backend) {
    return Backend.SQL_UPDATE + getTableName() + Backend.SQL_SET +
           CN_PDOCLASSID + Backend.SQL_EQUAL_PAR_COMMA +
           CN_LOCKEDBY + Backend.SQL_EQUAL_PAR_COMMA +
           CN_LOCKEDSINCE + Backend.SQL_EQUAL_PAR_COMMA +
           CN_LOCKEXPIRY + Backend.SQL_EQUAL_PAR_COMMA +
           CN_SERIAL + Backend.SQL_EQUAL + CN_SERIAL + Backend.SQL_PLUS_ONE +
           Backend.SQL_WHERE + CN_ID + Backend.SQL_EQUAL_PAR +
           Backend.SQL_AND + CN_SERIAL + Backend.SQL_EQUAL_PAR;
  }

  /**
   * Gets the attribute pdoClassId.
   *
   * @return class ID of the PDO
   */
  @Persistent(ordinal=0, comment="class ID of the PDO")
  public int getPdoClassId()    {
    return pdoClassId;
  }

  /**
   * Sets the attribute pdoClassId.
   *
   * @param pdoClassId class ID of the PDO
   */
  public void setPdoClassId(int pdoClassId) {
    assertMutable();
    this.pdoClassId = pdoClassId;
  }

  /**
   * Gets the attribute lockedBy.
   *
   * @return userId of token lock holder
   */
  @Persistent(ordinal=1, comment="userId of token lock holder")
  public long getLockedBy()    {
    return lockedBy;
  }

  /**
   * Sets the attribute lockedBy.
   *
   * @param lockedBy userId of token lock holder
   */
  public void setLockedBy(long lockedBy) {
    assertMutable();
    this.lockedBy = lockedBy;
  }

  /**
   * Gets the attribute lockedSince.
   *
   * @return time since token lock given to user
   */
  @Persistent(ordinal=2, comment="time since token lock given to user")
  public Timestamp getLockedSince()    {
    return lockedSince;
  }

  /**
   * Sets the attribute lockedSince.
   *
   * @param lockedSince time since token lock given to user
   */
  public void setLockedSince(Timestamp lockedSince) {
    Timestamp.setUTC(lockedSince, false);
    Freezable.freeze(lockedSince);
    assertMutable();
    this.lockedSince = lockedSince;
  }

  /**
   * Gets the attribute lockExpiry.
   *
   * @return time when token lock expires
   */
  @Persistent(ordinal=3, comment="time when token lock expires")
  public Timestamp getLockExpiry()    {
    return lockExpiry;
  }

  /**
   * Sets the attribute lockExpiry.
   *
   * @param lockExpiry time when token lock expires
   */
  public void setLockExpiry(Timestamp lockExpiry) {
    Timestamp.setUTC(lockExpiry, false);
    Freezable.freeze(lockExpiry);
    assertMutable();
    this.lockExpiry = lockExpiry;
  }

  //</editor-fold>//GEN-END:methods


  @Override
  public long getPdoId() {
    return getId();
  }

  /**
   * Creates a diagnostic string for logging.
   *
   * @return the diagnostic string
   */
  public String toDiagnosticString() {
    return "classId " + pdoClassId +
           ", id " + getPdoId() +
           ", locked by " + lockedBy +
           ", since " + lockedSince +
           ", expiry " + lockExpiry;
  }

}
