/*
 * Decompiled with CFR 0.152.
 */
package org.tentackle.persist;

import java.lang.ref.WeakReference;
import org.tentackle.common.Timestamp;
import org.tentackle.daemon.Scavenger;
import org.tentackle.log.Logger;
import org.tentackle.log.LoggerFactory;
import org.tentackle.pdo.PersistenceException;
import org.tentackle.pdo.Session;
import org.tentackle.pdo.SessionClosedException;
import org.tentackle.pdo.SessionInfo;
import org.tentackle.pdo.SessionPool;
import org.tentackle.persist.ConnectionManager;
import org.tentackle.persist.Db;
import org.tentackle.persist.ManagedConnection;

public class DefaultDbPool
implements SessionPool {
    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultDbPool.class);
    private static final long SECOND = 1000L;
    private static final long MINUTE = 60000L;
    private static final long SLEEP_INTERVAL = 10000L;
    private final String name;
    private final ConnectionManager conMgr;
    private final SessionInfo sessionInfo;
    private final int iniSize;
    private int incSize;
    private int minSize;
    private int maxSize;
    private long maxIdleMinutes;
    private long maxUsageMinutes;
    private boolean initialized;
    private PooledDb[] pool;
    private int[] freeList;
    private int freeCount;
    private int[] unusedList;
    private int unusedCount;
    private Thread timeoutThread;
    private boolean shutdownRequested;

    public DefaultDbPool(String name, ConnectionManager conMgr, SessionInfo sessionInfo, int iniSize, int incSize, int minSize, int maxSize, long maxIdleMinutes, long maxUsageMinutes) {
        if (maxSize > 0 && (maxSize < iniSize || maxSize < minSize) || minSize < 1 || incSize < 1 || iniSize < 1) {
            throw new IllegalArgumentException("illegal size parameters");
        }
        if (name == null) {
            throw new NullPointerException("name must not be null");
        }
        this.name = name;
        this.conMgr = conMgr;
        this.sessionInfo = sessionInfo;
        this.iniSize = iniSize;
        this.incSize = incSize;
        this.minSize = minSize;
        this.maxSize = maxSize;
        this.maxIdleMinutes = maxIdleMinutes;
        this.maxUsageMinutes = maxUsageMinutes;
        this.pool = new PooledDb[iniSize];
        this.freeList = new int[iniSize];
        this.unusedList = new int[iniSize];
        for (int i = 0; i < iniSize; ++i) {
            this.pool[i] = null;
            this.freeList[this.freeCount++] = i;
            this.unusedList[i] = -1;
        }
    }

    public DefaultDbPool(ConnectionManager conMgr, SessionInfo ui) {
        this("default-pool", conMgr, ui, 8, 2, 4, conMgr.getMaxSessions(), 60L, 1440L);
    }

    public String toString() {
        return this.name;
    }

    public String getName() {
        return this.name;
    }

    public SessionInfo getSessionInfo() {
        return this.sessionInfo;
    }

    public ConnectionManager getConnectionManager() {
        return this.conMgr;
    }

    public int getIniSize() {
        return this.iniSize;
    }

    public synchronized int getIncSize() {
        return this.incSize;
    }

    public synchronized void setIncSize(int incSize) {
        if (incSize < 1) {
            throw new IllegalArgumentException("increment size must be at least 1");
        }
        this.incSize = incSize;
    }

    public synchronized int getMinSize() {
        return this.minSize;
    }

    public synchronized void setMinSize(int minSize) {
        if (minSize < 1) {
            throw new IllegalArgumentException("minimum size must be at least 1");
        }
        this.minSize = minSize;
    }

    public synchronized int getMaxSize() {
        return this.maxSize;
    }

    public synchronized void setMaxSize(int maxSize) {
        if (maxSize < this.minSize) {
            throw new IllegalArgumentException("maximum size must not be lower than minsize=" + this.minSize);
        }
        this.maxSize = maxSize;
    }

    public synchronized long getMaxIdleMinutes() {
        return this.maxIdleMinutes;
    }

    public synchronized void setMaxIdleMinutes(long maxIdleMinutes) {
        this.maxIdleMinutes = maxIdleMinutes;
    }

    public synchronized long getMaxUsageMinutes() {
        return this.maxUsageMinutes;
    }

    public synchronized void setMaxUsageMinutes(long maxUsageMinutes) {
        this.maxUsageMinutes = maxUsageMinutes;
    }

    private void createDbInstances(int num) {
        if (num > this.freeCount) {
            int nSize = this.pool.length + num - this.freeCount;
            if (this.maxSize > 0 && nSize > this.maxSize) {
                nSize = this.maxSize;
            }
            if (nSize <= this.pool.length) {
                throw new PersistenceException("cannot create more Db instances, max. poolsize " + this.maxSize + " reached");
            }
            PooledDb[] nPool = new PooledDb[nSize];
            int[] nFreeList = new int[nSize];
            int[] nUnusedList = new int[nSize];
            System.arraycopy(this.pool, 0, nPool, 0, this.pool.length);
            System.arraycopy(this.freeList, 0, nFreeList, 0, this.pool.length);
            System.arraycopy(this.unusedList, 0, nUnusedList, 0, this.pool.length);
            for (int i = this.pool.length; i < nSize; ++i) {
                nPool[i] = null;
                nFreeList[this.freeCount++] = i;
                nUnusedList[i] = -1;
            }
            this.pool = nPool;
            this.freeList = nFreeList;
            this.unusedList = nUnusedList;
        }
        while (num > 0 && this.freeCount > 0) {
            int index = this.freeList[this.freeCount - 1];
            this.pool[index] = new PooledDb();
            --this.freeCount;
            this.unusedList[this.unusedCount++] = index;
            --num;
        }
    }

    private void removeDbInstance(Db dbToRemove, int index) {
        Db db = (Db)this.pool[index].refDb.get();
        if (db != dbToRemove) {
            throw new PersistenceException(dbToRemove + " to remove does not match " + db + " in pool " + this + " at index " + index);
        }
        if (db != null) {
            try {
                if (db.isOpen()) {
                    db.close();
                }
                db.setPoolId(-1);
            }
            catch (RuntimeException re) {
                LOGGER.severe("closing pooled db failed", (Throwable)re);
            }
        }
        this.removeDbIndex(index);
    }

    private void removeDbIndex(int index) {
        this.pool[index] = null;
        this.freeList[this.freeCount++] = index;
        for (int i = 0; i < this.unusedCount; ++i) {
            if (this.unusedList[i] != index) continue;
            System.arraycopy(this.unusedList, i + 1, this.unusedList, i, this.unusedCount - (i + 1));
            --this.unusedCount;
            break;
        }
    }

    protected void closeDb(Db db) {
        db.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        this.shutdownRequested = true;
        if (this.timeoutThread != null && this.timeoutThread.isAlive()) {
            this.timeoutThread.interrupt();
            try {
                this.timeoutThread.join();
            }
            catch (InterruptedException ex) {
                LOGGER.warning("shutdown " + this.timeoutThread + " for " + this + " failed", (Throwable)ex);
            }
        }
        DefaultDbPool defaultDbPool = this;
        synchronized (defaultDbPool) {
            for (PooledDb pdb : this.pool) {
                if (pdb == null) continue;
                pdb.close();
            }
            this.pool = null;
            this.freeList = null;
            this.unusedList = null;
            this.timeoutThread = null;
        }
    }

    public synchronized boolean isShutdown() {
        return this.pool == null;
    }

    public synchronized int getSize() {
        return this.pool == null ? 0 : this.pool.length - this.freeCount;
    }

    public synchronized Db getSession() {
        int poolId;
        Db db;
        this.assertNotShutdown();
        if (!this.initialized) {
            this.createDbInstances(this.iniSize);
            this.timeoutThread = new TimeoutThread();
            this.timeoutThread.start();
            this.initialized = true;
        }
        if (this.unusedCount == 0) {
            this.createDbInstances(this.incSize);
        }
        if (!(db = this.pool[poolId = this.unusedList[--this.unusedCount]].db).isOpen()) {
            throw new SessionClosedException((Session)db, this + ": Db has been closed unexpectedly");
        }
        this.pool[poolId].use(Thread.currentThread());
        db.setPoolId(poolId + 1);
        LOGGER.fine("{0}: Db {1} assigned to pool id {2}", new Object[]{this, db, poolId});
        return db;
    }

    public synchronized void putSession(Session session) {
        this.assertNotShutdown();
        Db db = (Db)session;
        if (db.getPool() != this) {
            throw new PersistenceException((Session)db, "Db is not managed by pool " + this);
        }
        int poolId = db.getPoolId();
        if (poolId < -1 || poolId > this.pool.length) {
            throw new PersistenceException((Session)db, this + ": Db has invalid poolid " + poolId);
        }
        if (db.isOpen()) {
            boolean txRunning = db.isTxRunning();
            if (txRunning) {
                db.rollbackImmediately();
            }
            if (poolId > 0) {
                ManagedConnection con = db.getConnection();
                if (con != null) {
                    con.closePreparedStatements(true);
                    this.removeDbInstance(db, poolId - 1);
                } else {
                    LOGGER.fine("{0}: Db {1} returned to pool, id {2}", new Object[]{this, db, poolId});
                    this.pool[--poolId].unUse(db);
                    this.unusedList[this.unusedCount++] = poolId;
                }
                db.setPoolId(0);
                db.setSessionGroupId(0);
                db.setOwnerThread(null);
            }
            if (txRunning) {
                throw new PersistenceException((Session)db, "Db was still running a transaction -- rolled back!");
            }
        } else if (poolId > 0) {
            this.removeDbInstance(db, poolId - 1);
            LOGGER.warning(this + ": returned Db " + db + " was closed and removed from pool", new Object[0]);
        } else {
            LOGGER.warning(this + ": returned Db " + db + " was closed and already returned to pool", new Object[0]);
        }
    }

    private void assertNotShutdown() {
        if (this.pool == null) {
            throw new PersistenceException(this + " already shutdown");
        }
    }

    private class TimeoutThread
    extends Thread
    implements Scavenger {
        private TimeoutThread() {
            super("Db-Pool '" + DefaultDbPool.this.getName() + "' Timeout Thread");
            this.setDaemon(true);
        }

        public boolean isScavenging() {
            return true;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            LOGGER.info(this + " started", new Object[0]);
            while (!DefaultDbPool.this.shutdownRequested) {
                try {
                    try {
                        TimeoutThread.sleep(10000L);
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                    if (DefaultDbPool.this.shutdownRequested) continue;
                    long curtime = System.currentTimeMillis();
                    DefaultDbPool defaultDbPool = DefaultDbPool.this;
                    synchronized (defaultDbPool) {
                        int i;
                        for (i = 0; i < DefaultDbPool.this.unusedCount; ++i) {
                            long usedMinutes;
                            boolean usageTimedOut;
                            boolean idleTimedOut;
                            int index = DefaultDbPool.this.unusedList[i];
                            PooledDb pooledDb = DefaultDbPool.this.pool[index];
                            long idleMinutes = pooledDb.idleMinutes(curtime);
                            boolean bl = idleTimedOut = idleMinutes > DefaultDbPool.this.getMaxIdleMinutes();
                            if (idleTimedOut) {
                                LOGGER.info("{0} idle for {1} (max={2}) -> closed", new Object[]{pooledDb, idleMinutes, DefaultDbPool.this.getMaxIdleMinutes()});
                            }
                            boolean bl2 = usageTimedOut = (usedMinutes = pooledDb.usedMinutes(curtime)) > DefaultDbPool.this.getMaxUsageMinutes();
                            if (usageTimedOut) {
                                LOGGER.info("{0} used for {1} (max={2}) -> closed", new Object[]{pooledDb, usedMinutes, DefaultDbPool.this.getMaxUsageMinutes()});
                            }
                            if (idleTimedOut || usageTimedOut) {
                                DefaultDbPool.this.removeDbInstance(pooledDb.db, index);
                                --i;
                                continue;
                            }
                            if (!pooledDb.db.isRemote()) continue;
                            try {
                                pooledDb.db.setAlive(true);
                                continue;
                            }
                            catch (RuntimeException ex) {
                                LOGGER.severe("remote keep alive failed", (Throwable)ex);
                                DefaultDbPool.this.removeDbInstance(pooledDb.db, index);
                                --i;
                            }
                        }
                        i = 0;
                        for (PooledDb pooledDb : DefaultDbPool.this.pool) {
                            if (pooledDb != null && pooledDb.isUnreferenced()) {
                                LOGGER.warning("unreferenced " + pooledDb + " last used by " + pooledDb.usingThreadStr + " in MDC{" + pooledDb.mdcStr + "} since " + new Timestamp(pooledDb.usedSince) + " -> removed from pool!", new Object[0]);
                                DefaultDbPool.this.removeDbIndex(i);
                            }
                            ++i;
                        }
                        int size = DefaultDbPool.this.getSize();
                        if (size < DefaultDbPool.this.getMinSize()) {
                            DefaultDbPool.this.createDbInstances(DefaultDbPool.this.getMinSize() - size);
                        }
                    }
                }
                catch (Exception ex) {
                    LOGGER.severe("cleaning up unused Db instance(s) failed", (Throwable)ex);
                }
            }
            LOGGER.info(this + " terminated", new Object[0]);
        }
    }

    private class PooledDb {
        private Db db;
        private final String dbStr;
        private final WeakReference<Db> refDb;
        private long usedSince;
        private String usingThreadStr;
        private String mdcStr;
        private long unusedSince;
        private long firstUse;

        private PooledDb() {
            LOGGER.fine("open pooled Db for {0}, connection manager {1}", new Object[]{DefaultDbPool.this.sessionInfo, DefaultDbPool.this.conMgr});
            this.db = new Db(DefaultDbPool.this.conMgr, DefaultDbPool.this.sessionInfo.clone());
            this.db.open();
            this.db.setPool(DefaultDbPool.this);
            this.dbStr = this.db.toString();
            this.unusedSince = System.currentTimeMillis();
            this.refDb = new WeakReference<Db>(this.db);
        }

        public String toString() {
            return this.dbStr;
        }

        private void close() {
            Db dbToClose = (Db)this.refDb.get();
            if (dbToClose != null) {
                try {
                    DefaultDbPool.this.closeDb(dbToClose);
                }
                catch (RuntimeException rex) {
                    LOGGER.warning("closing pooled Db " + dbToClose + " failed", (Throwable)rex);
                }
                finally {
                    this.db = null;
                }
            }
        }

        private void use(Thread usingThread) {
            if (this.db == null) {
                throw new PersistenceException("unexpected loss of reference to " + this.dbStr + " (last returned by " + this.usingThreadStr + " since " + new Timestamp(this.unusedSince) + ")");
            }
            this.usedSince = System.currentTimeMillis();
            if (this.firstUse == 0L) {
                this.firstUse = this.usedSince;
            }
            this.unusedSince = 0L;
            this.usingThreadStr = usingThread.toString();
            this.mdcStr = LOGGER.getMappedDiagnosticContext().toString();
            this.db = null;
        }

        private void unUse(Db db) {
            this.db = (Db)this.refDb.get();
            if (this.db == null) {
                throw new PersistenceException("unexpected loss of reference to " + this.dbStr + " (last use by " + this.usingThreadStr + " since " + new Timestamp(this.usedSince) + ")");
            }
            if (this.db != db) {
                this.db = null;
                throw new PersistenceException("attempt to unuse " + db + " in wrong slot " + this.dbStr + " (last use by " + this.usingThreadStr + " since " + new Timestamp(this.usedSince) + ")");
            }
            this.unusedSince = System.currentTimeMillis();
            this.usedSince = 0L;
            this.usingThreadStr = null;
            this.mdcStr = null;
        }

        private boolean isUnreferenced() {
            return this.refDb.get() == null;
        }

        private long idleMinutes(long currentTimeMillis) {
            return this.firstUse == 0L || this.unusedSince == 0L ? 0L : (currentTimeMillis - this.unusedSince) / 60000L;
        }

        private long usedMinutes(long currentTimeMillis) {
            return this.firstUse == 0L ? 0L : (currentTimeMillis - this.firstUse) / 60000L;
        }
    }
}

