/**
 * Tentackle - http://www.tentackle.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */



package org.tentackle.persist;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import org.tentackle.pdo.ClassId;
import org.tentackle.pdo.ModificationEvent;
import org.tentackle.pdo.ModificationListenerAdapter;
import org.tentackle.pdo.PdoTracker;
import org.tentackle.pdo.PersistenceException;
import org.tentackle.pdo.TableName;
import org.tentackle.persist.rmi.DbPreferencesNodeRemoteDelegate;
import org.tentackle.prefs.PersistedPreferencesFactory;
import org.tentackle.sql.Backend;



/**
 * @> $mapfile
 *
 * # a preferences node
 * name := $classname
 * id := $classid
 * table := $tablename
 *
 * ## attributes
 * [remote, tableserial]
 *
 * String(32)     user            username        name of user, null if system [mapnull]
 * String(192)    name            nodename        name of the node
 * long           parentId        parentid        ID of parent node, 0 if rootnode
 *
 * ## indexes
 * unique index nodename := nodename, username
 * index parentid := parentid
 * index tableserial := tableserial
 *
 * @<
 */


/**
 * Preferences Node stored in the database.
 *
 * @author harald
 */
@ClassId(/**/3/**/)   // @wurblet < Inject $classid
@TableName(/**/"prefnode"/**/)   // @wurblet < Inject --string $tablename
public class DbPreferencesNode extends AbstractDbObject<DbPreferencesNode> {

  private static final long serialVersionUID = 4677525106428955014L;

  /** the database tablename. */
  public static final String TABLENAME = /**/"prefnode"/**/;  // @wurblet < Inject --string $tablename


  private static final DbObjectClassVariables<DbPreferencesNode> CLASSVARIABLES =
    new DbObjectClassVariables<>(DbPreferencesNode.class,
                                 /**/3/**/,  // @wurblet < Inject $classid
                                 TABLENAME);



  static {
    // register updater for sync between jvms
    PdoTracker.getInstance().addModificationListener(new ModificationListenerAdapter(TABLENAME) {
      @Override
      public void dataChanged(ModificationEvent ev) {
        DbPreferencesFactory factory = (DbPreferencesFactory) PersistedPreferencesFactory.getInstance();
        if (factory.isAutoSync()) {
          factory.expireNodes(ev.getSerial());
        }
      }
    });
  }





  /**
   * Creates a node.
   *
   * @param db the db connection
   */
  public DbPreferencesNode (Db db)    {
    super(db);
  }

  /**
   * Creates a node (without db).
   */
  public DbPreferencesNode() {
    super();
  }



  @Override
  public boolean isCountingModification(char modType) {
    return true;
  }


  @Override
  public DbObjectClassVariables<DbPreferencesNode> getClassVariables() {
    return CLASSVARIABLES;
  }



  /**
   * Gets all nodes belonging to a parent node.
   *
   * @param parentId the ID of the parent node, 0 = rootnode
   * @return List of nodes
   *
   * @wurblet selectByParentId DbSelectList --model=$mapfile parentId
   */
  // <editor-fold defaultstate="collapsed" desc=" Code generated by wurblet. Do not edit! ">//GEN-BEGIN:selectByParentId

  public List<DbPreferencesNode> selectByParentId(long parentId) {
    if (getSession().isRemote())  {
      // invoke remote method
      try {
        List<DbPreferencesNode> list = ((DbPreferencesNodeRemoteDelegate) getRemoteDelegate()).
                selectByParentId(parentId);
        getSession().applyTo(list);
        return list;
      }
      catch (RemoteException e) {
        throw PersistenceException.createFromRemoteException(getSession(), e);
      }
    }
    // else: local mode
    PreparedStatementWrapper st = getPreparedStatement(SELECT_BY_PARENT_ID_STMT,
      () -> {
        StringBuilder sql = createSelectAllInnerSql();
        sql.append(Backend.SQL_AND);
        sql.append(CN_PARENTID);
        sql.append(Backend.SQL_EQUAL_PAR);
        getBackend().buildSelectSql(sql, false, 0, 0);
        return sql.toString();
      }
    );
    int ndx = 1;
    st.setLong(ndx++, parentId);
    try (ResultSetWrapper rs = st.executeQuery()) {
      List<DbPreferencesNode> list = new ArrayList<>();
      boolean derived = getClass() != DbPreferencesNode.class;
      while (rs.next()) {
        DbPreferencesNode obj = derived ? newInstance() : new DbPreferencesNode(getSession());
        obj = obj.readFromResultSetWrapper(rs);
        if (obj != null)  {
          list.add(obj);
        }
      }
      return list;
    }
  }

  private static final StatementId SELECT_BY_PARENT_ID_STMT = new StatementId();


  // </editor-fold>//GEN-END:selectByParentId




  /**
   * Selects the object IDs of all childs nodes.
   *
   * @return the IDs
   */
  public Set<Long> selectChildIds() {
    if (getSession().isRemote())  {
      try {
        return ((DbPreferencesNodeRemoteDelegate) getRemoteDelegate()).selectChildIds(getId());
      }
      catch (RemoteException e) {
        throw PersistenceException.createFromRemoteException(this, e);
      }
    }
    else  {
      PreparedStatementWrapper st = getPreparedStatement(selectChildIdsStatementId,
        () -> "SELECT " + CN_ID + " FROM " + TABLENAME +
                " WHERE " + CN_PARENTID + "=?"
      );
      st.setLong(1, getId());
      Set<Long> set = new TreeSet<>();
      try (ResultSetWrapper rs = st.executeQuery()) {
        while (rs.next()) {
          set.add(rs.getALong(1));
        }
      }
      return set;
    }
  }

  private static final StatementId selectChildIdsStatementId = new StatementId();



  /**
   * Selects a node by user and name.
   *
   * @param user is the username, null = system
   * @param name is the absolute pathname of the node, ""=root
   * @return the node
   *
   * @wurblet selectByUserAndName DbSelectUnique --model=$mapfile user name
   */
  // <editor-fold defaultstate="collapsed" desc=" Code generated by wurblet. Do not edit! ">//GEN-BEGIN:selectByUserAndName

  public DbPreferencesNode selectByUserAndName(String user, String name) {
    if (getSession().isRemote())  {
      // invoke remote method
      try {
        DbPreferencesNode obj = ((DbPreferencesNodeRemoteDelegate) getRemoteDelegate()).
                selectByUserAndName(user, name);
        getSession().applyTo(obj);
        return obj;
      }
      catch (RemoteException e) {
        throw PersistenceException.createFromRemoteException(this, e);
      }
    }
    // else: local mode
    PreparedStatementWrapper st = getPreparedStatement(SELECT_BY_USER_AND_NAME_STMT,
      () -> {
        StringBuilder sql = createSelectAllInnerSql();
        sql.append(Backend.SQL_AND);
        sql.append(CN_USER);
        sql.append(Backend.SQL_EQUAL_PAR);
        sql.append(Backend.SQL_AND);
        sql.append(CN_NAME);
        sql.append(Backend.SQL_EQUAL_PAR);
        getBackend().buildSelectSql(sql, false, 0, 0);
        return sql.toString();
      }
    );
    int ndx = 1;
    st.setString(ndx++, user, true);
    st.setString(ndx++, name);
    try (ResultSetWrapper rs = st.executeQuery()) {
      if (rs.next()) {
        return readFromResultSetWrapper(rs);
      }
      return null;  // not found
    }
  }

  private static final StatementId SELECT_BY_USER_AND_NAME_STMT = new StatementId();


  // </editor-fold>//GEN-END:selectByUserAndName







  // generated GETTER/SETTER

  // @wurblet methods MethodsImpl --noif --model=$mapfile

  // <editor-fold defaultstate="collapsed" desc=" Code generated by wurblet. Do not edit! ">//GEN-BEGIN:methods


  /**
   * Object provides the tableserial.
   * @return true
   */
  @Override
  public boolean isTableSerialProvided() {
    return true;
  }

  @Override
  public void getFields(ResultSetWrapper rs) {
    super.getFields(rs);
    if (rs.configureSection(CLASSVARIABLES)) {
      rs.configureColumn(CN_TABLESERIAL);
      rs.configureColumn(CN_USER);
      rs.configureColumn(CN_NAME);
      rs.configureColumn(CN_PARENTID);
      rs.configureColumn(CN_ID);
      rs.configureColumn(CN_SERIAL);
    }
    if (rs.getRow() <= 0) {
      throw new PersistenceException(getSession(), "no valid row");
    }
    setTableSerial(rs.getLong());
    user = rs.getString(true);
    name = rs.getString();
    parentId = rs.getLong();
    setId(rs.getLong());
    setSerial(rs.getLong());
  }

  @Override
  public int setFields(PreparedStatementWrapper st) {
    int ndx = super.setFields(st);
    st.setLong(++ndx, getTableSerial());
    st.setString(++ndx, user, true);
    st.setString(++ndx, name);
    st.setLong(++ndx, parentId);
    st.setLong(++ndx, getId());
    st.setLong(++ndx, getSerial());
    return ndx;
  }

  @Override
  public String createInsertSql() {
    return Backend.SQL_INSERT_INTO + getTableName() + Backend.SQL_LEFT_PARENTHESIS +
           CN_TABLESERIAL + Backend.SQL_COMMA +
           CN_USER + Backend.SQL_COMMA +
           CN_NAME + Backend.SQL_COMMA +
           CN_PARENTID + Backend.SQL_COMMA +
           CN_ID + Backend.SQL_COMMA +
           CN_SERIAL +
           Backend.SQL_INSERT_VALUES +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR + Backend.SQL_RIGHT_PARENTHESIS;
  }

  @Override
  public String createUpdateSql() {
    return Backend.SQL_UPDATE + getTableName() + Backend.SQL_SET +
           CN_TABLESERIAL + Backend.SQL_EQUAL_PAR_COMMA +
           CN_USER + Backend.SQL_EQUAL_PAR_COMMA +
           CN_NAME + Backend.SQL_EQUAL_PAR_COMMA +
           CN_PARENTID + Backend.SQL_EQUAL_PAR_COMMA +
           CN_SERIAL + Backend.SQL_EQUAL + CN_SERIAL + Backend.SQL_PLUS_ONE +
           Backend.SQL_WHERE + CN_ID + Backend.SQL_EQUAL_PAR +
           Backend.SQL_AND + CN_SERIAL + Backend.SQL_EQUAL_PAR;
  }

  /**
   * Gets the attribute user.
   *
   * @return name of user, null if system
   */
  public String getUser()    {
    return user;
  }

  /**
   * Sets the attribute user.
   *
   * @param user name of user, null if system
   */
  public void setUser(String user) {
    assertMutable();
    this.user = user;
  }

  /**
   * Gets the attribute name.
   *
   * @return name of the node
   */
  public String getName()    {
    return name;
  }

  /**
   * Sets the attribute name.
   *
   * @param name name of the node
   */
  public void setName(String name) {
    assertMutable();
    this.name = name;
  }

  /**
   * Gets the attribute parentId.
   *
   * @return ID of parent node, 0 if rootnode
   */
  public long getParentId()    {
    return parentId;
  }

  /**
   * Sets the attribute parentId.
   *
   * @param parentId ID of parent node, 0 if rootnode
   */
  public void setParentId(long parentId) {
    assertMutable();
    this.parentId = parentId;
  }

  // </editor-fold>//GEN-END:methods



  /**
   * Gets the basename of the node.
   *
   * @return the node's basename
   */
  public String getBaseName() {
    int lastSlash = name.lastIndexOf('/');
    return lastSlash >= 0 ? name.substring(lastSlash + 1) : name;
  }


  @Override
  public String toString()  {
    return "ID=" + getId() + ", parentId=" + parentId +
           (user == null ? ", <system>" : (", user='" + user + "'")) +
           ", name='" + name + "'";
  }




  // record members
  // @wurblet declare Declare --model=$mapfile

  // <editor-fold defaultstate="collapsed" desc=" Code generated by wurblet. Do not edit! ">//GEN-BEGIN:declare


  /** name of user, null if system. */
  private String user;

  /** name of the node. */
  private String name;

  /** ID of parent node, 0 if rootnode. */
  private long parentId;

  // </editor-fold>//GEN-END:declare


  // @wurblet fieldNames ColumnNames --model=$mapfile

  // <editor-fold defaultstate="collapsed" desc=" Code generated by wurblet. Do not edit! ">//GEN-BEGIN:fieldNames


  /** database column name for 'user'. */
  public static final String CN_USER = "username";

  /** database column name for 'name'. */
  public static final String CN_NAME = "nodename";

  /** database column name for 'parentId'. */
  public static final String CN_PARENTID = "parentid";

  // </editor-fold>//GEN-END:fieldNames

}

