/*
 * Tentackle - https://tentackle.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package org.tentackle.persist.security;

import org.tentackle.dbms.ModificationType;
import org.tentackle.dbms.PreparedStatementWrapper;
import org.tentackle.dbms.ResultSetWrapper;
import org.tentackle.dbms.StatementId;
import org.tentackle.log.Logger;
import org.tentackle.misc.TrackedList;
import org.tentackle.pdo.DomainContext;
import org.tentackle.pdo.Pdo;
import org.tentackle.pdo.PdoCache;
import org.tentackle.pdo.PersistentDomainObject;
import org.tentackle.pdo.PersistentObjectService;
import org.tentackle.persist.AbstractPersistentObject;
import org.tentackle.persist.PersistentObjectClassVariables;
import org.tentackle.persist.security.rmi.SecurityRemoteDelegate;
import org.tentackle.security.pdo.Security;
import org.tentackle.security.pdo.SecurityPersistence;
import org.tentackle.session.PersistenceException;
import org.tentackle.session.Session;
import org.tentackle.session.SessionUtilities;
import org.tentackle.sql.Backend;

import java.io.Serial;
import java.rmi.RemoteException;
import java.util.List;
import java.util.Objects;



/**
 * Security rule persistence implementation.
 *
 * @author harald
 */
@PersistentObjectService(Security.class)
public class SecurityPersistenceImpl extends AbstractPersistentObject<Security, SecurityPersistenceImpl> implements SecurityPersistence {

  @Serial
  private static final long serialVersionUID = 1L;

  private static final Logger LOGGER = Logger.get(SecurityPersistenceImpl.class);


  // @wurblet classVariables ClassVariables

  //<editor-fold defaultstate="collapsed" desc="code 'classVariables' generated by wurblet ClassVariables">//GEN-BEGIN:classVariables

  /** Variables common to all instances of SecurityPersistenceImpl. */
  public static final PersistentObjectClassVariables<Security, SecurityPersistenceImpl> CLASSVARIABLES =
            PersistentObjectClassVariables.create(
                    Security.class,
                    SecurityPersistenceImpl.class,
                    "scrty",
                    null,
                    null);

  @Override
  public PersistentObjectClassVariables<Security, SecurityPersistenceImpl> getClassVariables() {
    return CLASSVARIABLES;
  }

  //</editor-fold>//GEN-END:classVariables

  // @wurblet fieldnames ColumnNames

  //<editor-fold defaultstate="collapsed" desc="code 'fieldnames' generated by wurblet ColumnNames">//GEN-BEGIN:fieldnames


  /** database column name for 'objectClassName'. */
  public static final String CN_OBJECTCLASSNAME = "objectclass";

  /** database column name for 'objectClassId'. */
  public static final String CN_OBJECTCLASSID = "objectclassid";

  /** database column name for 'objectId'. */
  public static final String CN_OBJECTID = "objectid";

  /** database column name for 'domainContextClassId'. */
  public static final String CN_DOMAINCONTEXTCLASSID = "contextclassid";

  /** database column name for 'domainContextId'. */
  public static final String CN_DOMAINCONTEXTID = "contextid";

  /** database column name for 'granteeClassId'. */
  public static final String CN_GRANTEECLASSID = "granteeclassid";

  /** database column name for 'granteeId'. */
  public static final String CN_GRANTEEID = "granteeid";

  /** database column name for 'priority'. */
  public static final String CN_PRIORITY = "secprio";

  /** database column name for 'permissions'. */
  public static final String CN_PERMISSIONS = "permissions";

  /** database column name for 'allowed'. */
  public static final String CN_ALLOWED = "allowed";

  /** database column name for 'message'. */
  public static final String CN_MESSAGE = "message";

  //</editor-fold>//GEN-END:fieldnames

  // @wurblet declare Declare

  //<editor-fold defaultstate="collapsed" desc="code 'declare' generated by wurblet Declare">//GEN-BEGIN:declare


  /** the protected classname, null if an entity. */
  private String objectClassName;

  /** the protected object's class id, 0 if not an entity. */
  private int objectClassId;

  /** the id of the protected object, 0 if all instances or not an entity. */
  private long objectId;

  /** the class id of the DomainContext's context entity, 0 if all contexts. */
  private int domainContextClassId;

  /** the id of DomainContext's context object, 0 if all instances. */
  private long domainContextId;

  /** the class id of the entity the permissions are granted to, null if all classes. */
  private int granteeClassId;

  /** the id of the entity the permissions are granted to, 0 if all grantees. */
  private long granteeId;

  /** the priority or evaluation order, 0 is highest or first. */
  private int priority;

  /** the permissions as a comma-separated list. */
  private String permissions;

  /** the false if denied, true if allowed. */
  private boolean allowed;

  /** the user message. */
  private String message;

  //</editor-fold>//GEN-END:declare


  /**
   * Creates a security rule.
   *
   * @param pdo the security PDO
   * @param context the domain context
   */
  public SecurityPersistenceImpl (Security pdo, DomainContext context)    {
    super(pdo, context);
  }

  /**
   * Creates a security rule with a session only.
   *
   * @param pdo the security PDO
   * @param session the session
   */
  public SecurityPersistenceImpl(Security pdo, Session session) {
    super(pdo, session);
  }

  /**
   * Creates a security rule without domain context or session.
   *
   * @param pdo the security PDO
   */
  public SecurityPersistenceImpl(Security pdo) {
    super(pdo);
  }

  /**
   * Creates a security rule without domain context or session.
   */
  public SecurityPersistenceImpl() {
    super();
  }

  /**
   * Gets the object this rule applies to.<br>
   * The object is retrieved in its valid context.
   * @return the object, null if none (class rule)
   */
  @Override
  public PersistentDomainObject<?> getObject() {
    if (objectId != 0 && objectClassId != 0)  {
      try {
        return Pdo.create(objectClassName, getDomainContext()).select(objectId);
      }
      catch (RuntimeException e) {
        LOGGER.warning("can't get secured object for " + objectClassName + ", " + objectId);
      }
    }
    return null;
  }

  @Override
  public void assertRootContextIsAccepted() {
    // always accepted
  }

  /**
   * Sets the object this rule applies to.<br>
   * Setting the null object also clears the objectClass
   * making the rule invalid.
   *
   * @param object the object, null to clear
   */
  @Override
  public void setObject(PersistentDomainObject<?> object)  {
    if (object == null) {
      setObjectId(0);
      setObjectClassName(null);
    }
    else  {
      setObjectId(object.getPersistenceDelegate().getId());
      setObjectClassName(object.getClass().getName());
    }
  }


  /**
   * Gets the grantee.
   *
   * @return the grantee, null if grant class
   */
  @Override
  public PersistentDomainObject<?> getGrantee() {
    return granteeId == 0 ? null : Pdo.create(SessionUtilities.getInstance().getClassName(
            granteeClassId), getDomainContext()).selectCached(granteeId);
  }


  /**
   * Sets the grantee.
   *
   * @param grantee the grantee, null to clear (all)
   */
  @Override
  public void setGrantee(PersistentDomainObject<?> grantee)  {
    if (grantee == null)  {
      setGranteeId(0);
      setGranteeClassId(0);
    }
    else  {
      setGranteeId(grantee.getPersistenceDelegate().getId());
      setGranteeClassId(grantee.getPersistenceDelegate().getClassId());
    }
  }


  /**
   * Gets the application context's object.<br>
   * The object will be loaded in its valid context.
   *
   * @return the context object, null if no context (all)
   */
  @Override
  public PersistentDomainObject<?> getDomainContextObject()  {
    return domainContextId == 0 ? null :
           Pdo.create(SessionUtilities.getInstance().getClassName(
                   domainContextClassId), getDomainContext()).selectCached(domainContextId);
  }


  /**
   * Sets the application context's object.
   *
   * @param contextObject the context object, null to clear (all)
   */
  @Override
  public void setDomainContextObject(PersistentDomainObject<?> contextObject)  {
    if (contextObject == null)  {
      setDomainContextId(0);
      setDomainContextClassId(0);
    }
    else {
      setDomainContextId(contextObject.getPersistenceDelegate().getId());
      setDomainContextClassId(contextObject.getPersistenceDelegate().getClassId());
    }
  }


  /**
   * for debugging only
   */
  @Override
  public String toString()  {
    return "Security ID=" + getId() + ", objectClass=" + objectClassName +
           ", objectId=" + objectId + ", contextId=" + domainContextId +
           ", contextClassId=" + domainContextClassId + ", granteeId=" + granteeId + ", granteeClass=" + granteeClassId +
           ", priority=" + priority + ", permissions='" + permissions +
           "', allowed=" + allowed + ", message='" + message + "'";
  }


  @Override
  public void assertRemoteSecurityManagerInitialized() {
    try {
      getRemoteDelegate().assertRemoteSecurityManagerInitialized(getDomainContext());
    }
    catch (RemoteException e) {
      throw PersistenceException.createFromRemoteException(this, e);
    }
  }

  @Override
  public Class<?> getObjectClass() {
    String className = objectClassId != 0 ?
      SessionUtilities.getInstance().getClassName(objectClassId) : objectClassName;
    try {
      return Class.forName(className);
    }
    catch (ClassNotFoundException ex) {
      throw new PersistenceException(this, ex);
    }
  }


  // @wurblet cache PdoCache --preload

  //<editor-fold defaultstate="collapsed" desc="code 'cache' generated by wurblet PdoCache">//GEN-BEGIN:cache

  /** Holder of the PDO cache singleton. */
  private static class CacheHolder {
    private static final PdoCache<Security> CACHE = createCache();

    private static PdoCache<Security> createCache() {
      PdoCache<Security> cache = Pdo.createPdoCache(Security.class, true, true, false);
      Pdo.listen(cache::expire, Security.class);
      return cache;
    }
  }

  @Override
  public PdoCache<Security> getCache() {
    return CacheHolder.CACHE;
  }

  @Override
  public boolean isCountingModification(ModificationType modType) {
    return true;
  }

  @Override
  public boolean isReadAllowed() {
    return true;
  }

  @Override
  public void expireCache(long maxSerial) {
    super.expireCache(maxSerial);
    CacheHolder.CACHE.expire(null, getTableName(), maxSerial);
  }

  @Override
  public Security selectCachedOnly(long id) {
    return getCache().select(getDomainContext(), id, false);
  }

  @Override
  public Security selectCached(long id) {
    return getCache().select(getDomainContext(), id);
  }

  @Override
  public List<Security> selectAllCached() {
    return getCache().selectAll(getDomainContext());
  }

  //</editor-fold>//GEN-END:cache


  // @wurblet selectByGrantee PdoSelectList --sort granteeClassId granteeId

  //<editor-fold defaultstate="collapsed" desc="code 'selectByGrantee' generated by wurblet PdoSelectList">//GEN-BEGIN:selectByGrantee


  private static final StatementId SELECT_BY_GRANTEE_STMT = new StatementId();

  @Override
  public List<Security> selectByGrantee(int granteeClassId, long granteeId) {
    if (getSession().isRemote())  {
      try {
        List<Security> list = getRemoteDelegate().selectByGrantee(getDomainContext(), granteeClassId, granteeId);
        configureRemoteObjects(getDomainContext(), list);
        return list;
      }
      catch (RemoteException e) {
        throw PersistenceException.createFromRemoteException(this, e);
      }
    }
    PreparedStatementWrapper st = getPreparedStatement(SELECT_BY_GRANTEE_STMT,
      () -> {
        StringBuilder sql = createSelectAllInnerSql();
        sql.append(Backend.SQL_AND);
        sql.append(getColumnName(CN_GRANTEECLASSID));
        sql.append(Backend.SQL_EQUAL_PAR);
        sql.append(Backend.SQL_AND);
        sql.append(getColumnName(CN_GRANTEEID));
        sql.append(Backend.SQL_EQUAL_PAR);
        getBackend().buildSelectSql(sql, false, 0, 0);
        String orderSuffix = orderBy();
        if (orderSuffix != null && !orderSuffix.isEmpty()) {
          sql.append(Backend.SQL_ORDERBY).append(orderSuffix);
        }
        return sql.toString();
      }
    );
    int ndx = 1;
    st.setInt(ndx++, granteeClassId);
    st.setLong(ndx, granteeId);
    return executeListQuery(st);
  }

  //</editor-fold>//GEN-END:selectByGrantee


  // @wurblet selectByObject PdoSelectList --sort --tracked objectClassId objectId

  //<editor-fold defaultstate="collapsed" desc="code 'selectByObject' generated by wurblet PdoSelectList">//GEN-BEGIN:selectByObject


  private static final StatementId SELECT_BY_OBJECT_STMT = new StatementId();

  @Override
  public TrackedList<Security> selectByObject(int objectClassId, long objectId) {
    if (getSession().isRemote())  {
      try {
        TrackedList<Security> list = getRemoteDelegate().selectByObject(getDomainContext(), objectClassId, objectId);
        configureRemoteObjects(getDomainContext(), list);
        return list;
      }
      catch (RemoteException e) {
        throw PersistenceException.createFromRemoteException(this, e);
      }
    }
    PreparedStatementWrapper st = getPreparedStatement(SELECT_BY_OBJECT_STMT,
      () -> {
        StringBuilder sql = createSelectAllInnerSql();
        sql.append(Backend.SQL_AND);
        sql.append(getColumnName(CN_OBJECTCLASSID));
        sql.append(Backend.SQL_EQUAL_PAR);
        sql.append(Backend.SQL_AND);
        sql.append(getColumnName(CN_OBJECTID));
        sql.append(Backend.SQL_EQUAL_PAR);
        getBackend().buildSelectSql(sql, false, 0, 0);
        String orderSuffix = orderBy();
        if (orderSuffix != null && !orderSuffix.isEmpty()) {
          sql.append(Backend.SQL_ORDERBY).append(orderSuffix);
        }
        return sql.toString();
      }
    );
    int ndx = 1;
    st.setInt(ndx++, objectClassId);
    st.setLong(ndx, objectId);
    return executeTrackedListQuery(st);
  }

  //</editor-fold>//GEN-END:selectByObject


  // @wurblet selectByObjectClass PdoSelectList --sort --tracked objectClassName

  //<editor-fold defaultstate="collapsed" desc="code 'selectByObjectClass' generated by wurblet PdoSelectList">//GEN-BEGIN:selectByObjectClass


  private static final StatementId SELECT_BY_OBJECT_CLASS_STMT = new StatementId();

  @Override
  public TrackedList<Security> selectByObjectClass(String objectClassName) {
    if (getSession().isRemote())  {
      try {
        TrackedList<Security> list = getRemoteDelegate().selectByObjectClass(getDomainContext(), objectClassName);
        configureRemoteObjects(getDomainContext(), list);
        return list;
      }
      catch (RemoteException e) {
        throw PersistenceException.createFromRemoteException(this, e);
      }
    }
    PreparedStatementWrapper st = getPreparedStatement(SELECT_BY_OBJECT_CLASS_STMT,
      () -> {
        StringBuilder sql = createSelectAllInnerSql();
        sql.append(Backend.SQL_AND);
        sql.append(getColumnName(CN_OBJECTCLASSNAME));
        sql.append(Backend.SQL_EQUAL_PAR);
        getBackend().buildSelectSql(sql, false, 0, 0);
        String orderSuffix = orderBy();
        if (orderSuffix != null && !orderSuffix.isEmpty()) {
          sql.append(Backend.SQL_ORDERBY).append(orderSuffix);
        }
        return sql.toString();
      }
    );
    int ndx = 1;
    st.setString(ndx, objectClassName);
    return executeTrackedListQuery(st);
  }

  //</editor-fold>//GEN-END:selectByObjectClass


  // @wurblet selectByDomainContext PdoSelectList --sort --tracked domainContextClassId domainContextId

  //<editor-fold defaultstate="collapsed" desc="code 'selectByDomainContext' generated by wurblet PdoSelectList">//GEN-BEGIN:selectByDomainContext


  private static final StatementId SELECT_BY_DOMAIN_CONTEXT_STMT = new StatementId();

  @Override
  public TrackedList<Security> selectByDomainContext(int domainContextClassId, long domainContextId) {
    if (getSession().isRemote())  {
      try {
        TrackedList<Security> list = getRemoteDelegate().selectByDomainContext(getDomainContext(), domainContextClassId, domainContextId);
        configureRemoteObjects(getDomainContext(), list);
        return list;
      }
      catch (RemoteException e) {
        throw PersistenceException.createFromRemoteException(this, e);
      }
    }
    PreparedStatementWrapper st = getPreparedStatement(SELECT_BY_DOMAIN_CONTEXT_STMT,
      () -> {
        StringBuilder sql = createSelectAllInnerSql();
        sql.append(Backend.SQL_AND);
        sql.append(getColumnName(CN_DOMAINCONTEXTCLASSID));
        sql.append(Backend.SQL_EQUAL_PAR);
        sql.append(Backend.SQL_AND);
        sql.append(getColumnName(CN_DOMAINCONTEXTID));
        sql.append(Backend.SQL_EQUAL_PAR);
        getBackend().buildSelectSql(sql, false, 0, 0);
        String orderSuffix = orderBy();
        if (orderSuffix != null && !orderSuffix.isEmpty()) {
          sql.append(Backend.SQL_ORDERBY).append(orderSuffix);
        }
        return sql.toString();
      }
    );
    int ndx = 1;
    st.setInt(ndx++, domainContextClassId);
    st.setLong(ndx, domainContextId);
    return executeTrackedListQuery(st);
  }

  //</editor-fold>//GEN-END:selectByDomainContext


  // @wurblet deleteByGrantee PdoDeleteBy granteeClassId granteeId

  //<editor-fold defaultstate="collapsed" desc="code 'deleteByGrantee' generated by wurblet PdoDeleteBy">//GEN-BEGIN:deleteByGrantee

  @Override
  public int deleteByGrantee(int granteeClassId, long granteeId) {
    if (getSession().isRemote())  {
      try {
        return getRemoteDelegate().deleteByGrantee(granteeClassId, granteeId);
      }
      catch (RemoteException e) {
        throw PersistenceException.createFromRemoteException(getSession(), e);
      }
    }
    // else: local mode
    // check security rules at class level
    if (!getClassVariables().isWriteAllowed(getDomainContext())) {
      throw new PersistenceException("deleteByGrantee denied by security rules");
    }
    PreparedStatementWrapper st = getPreparedStatement(DELETE_BY_GRANTEE_STMT,
      () -> {
        StringBuilder sql = createDeleteAllSql();
        sql.append(Backend.SQL_AND);
        sql.append(CN_GRANTEECLASSID);
        sql.append(Backend.SQL_EQUAL_PAR);
        sql.append(Backend.SQL_AND);
        sql.append(CN_GRANTEEID);
        sql.append(Backend.SQL_EQUAL_PAR);
        return sql.toString();
      }
    );
    int ndx = 1;
    st.setInt(ndx++, granteeClassId);
    st.setLong(ndx, granteeId);
    return st.executeUpdate();
  }

  private static final StatementId DELETE_BY_GRANTEE_STMT = new StatementId();


  //</editor-fold>//GEN-END:deleteByGrantee


  // @wurblet deleteByObject PdoDeleteBy objectClassName:NULL objectClassId objectId

  //<editor-fold defaultstate="collapsed" desc="code 'deleteByObject' generated by wurblet PdoDeleteBy">//GEN-BEGIN:deleteByObject

  @Override
  public int deleteByObject(int objectClassId, long objectId) {
    if (getSession().isRemote())  {
      try {
        return getRemoteDelegate().deleteByObject(objectClassId, objectId);
      }
      catch (RemoteException e) {
        throw PersistenceException.createFromRemoteException(getSession(), e);
      }
    }
    // else: local mode
    // check security rules at class level
    if (!getClassVariables().isWriteAllowed(getDomainContext())) {
      throw new PersistenceException("deleteByObject denied by security rules");
    }
    PreparedStatementWrapper st = getPreparedStatement(DELETE_BY_OBJECT_STMT,
      () -> {
        StringBuilder sql = createDeleteAllSql();
        sql.append(Backend.SQL_AND);
        sql.append(CN_OBJECTCLASSNAME);
        sql.append(Backend.SQL_ISNULL);
        sql.append(Backend.SQL_AND);
        sql.append(CN_OBJECTCLASSID);
        sql.append(Backend.SQL_EQUAL_PAR);
        sql.append(Backend.SQL_AND);
        sql.append(CN_OBJECTID);
        sql.append(Backend.SQL_EQUAL_PAR);
        return sql.toString();
      }
    );
    int ndx = 1;
    st.setInt(ndx++, objectClassId);
    st.setLong(ndx, objectId);
    return st.executeUpdate();
  }

  private static final StatementId DELETE_BY_OBJECT_STMT = new StatementId();


  //</editor-fold>//GEN-END:deleteByObject


  // @wurblet deleteByObjectClass PdoDeleteBy objectClassId:=:0 objectClassName

  //<editor-fold defaultstate="collapsed" desc="code 'deleteByObjectClass' generated by wurblet PdoDeleteBy">//GEN-BEGIN:deleteByObjectClass

  @Override
  public int deleteByObjectClass(String objectClassName) {
    if (getSession().isRemote())  {
      try {
        return getRemoteDelegate().deleteByObjectClass(objectClassName);
      }
      catch (RemoteException e) {
        throw PersistenceException.createFromRemoteException(getSession(), e);
      }
    }
    // else: local mode
    // check security rules at class level
    if (!getClassVariables().isWriteAllowed(getDomainContext())) {
      throw new PersistenceException("deleteByObjectClass denied by security rules");
    }
    PreparedStatementWrapper st = getPreparedStatement(DELETE_BY_OBJECT_CLASS_STMT,
      () -> {
        StringBuilder sql = createDeleteAllSql();
        sql.append(Backend.SQL_AND);
        sql.append(CN_OBJECTCLASSID);
        sql.append(Backend.SQL_EQUAL_PAR);
        sql.append(Backend.SQL_AND);
        sql.append(CN_OBJECTCLASSNAME);
        sql.append(Backend.SQL_EQUAL_PAR);
        return sql.toString();
      }
    );
    int ndx = 1;
    st.setInt(ndx++, 0);
    st.setString(ndx, objectClassName);
    return st.executeUpdate();
  }

  private static final StatementId DELETE_BY_OBJECT_CLASS_STMT = new StatementId();


  //</editor-fold>//GEN-END:deleteByObjectClass


  // @wurblet deleteByDomainContext PdoDeleteBy domainContextClassId domainContextId

  //<editor-fold defaultstate="collapsed" desc="code 'deleteByDomainContext' generated by wurblet PdoDeleteBy">//GEN-BEGIN:deleteByDomainContext

  @Override
  public int deleteByDomainContext(int domainContextClassId, long domainContextId) {
    if (getSession().isRemote())  {
      try {
        return getRemoteDelegate().deleteByDomainContext(domainContextClassId, domainContextId);
      }
      catch (RemoteException e) {
        throw PersistenceException.createFromRemoteException(getSession(), e);
      }
    }
    // else: local mode
    // check security rules at class level
    if (!getClassVariables().isWriteAllowed(getDomainContext())) {
      throw new PersistenceException("deleteByDomainContext denied by security rules");
    }
    PreparedStatementWrapper st = getPreparedStatement(DELETE_BY_DOMAIN_CONTEXT_STMT,
      () -> {
        StringBuilder sql = createDeleteAllSql();
        sql.append(Backend.SQL_AND);
        sql.append(CN_DOMAINCONTEXTCLASSID);
        sql.append(Backend.SQL_EQUAL_PAR);
        sql.append(Backend.SQL_AND);
        sql.append(CN_DOMAINCONTEXTID);
        sql.append(Backend.SQL_EQUAL_PAR);
        return sql.toString();
      }
    );
    int ndx = 1;
    st.setInt(ndx++, domainContextClassId);
    st.setLong(ndx, domainContextId);
    return st.executeUpdate();
  }

  private static final StatementId DELETE_BY_DOMAIN_CONTEXT_STMT = new StatementId();


  //</editor-fold>//GEN-END:deleteByDomainContext



  // @wurblet methods MethodsImpl --tracked

  //<editor-fold defaultstate="collapsed" desc="code 'methods' generated by wurblet MethodsImpl">//GEN-BEGIN:methods


  @Override
  public SecurityRemoteDelegate getRemoteDelegate() {
    return (SecurityRemoteDelegate) super.getRemoteDelegate();
  }

  @Override
  public boolean isRootEntity() {
    return true;
  }

  @Override
  public boolean isTracked() {
    return true;    // invoking isModified() is ok
  }

  @Override
  public String orderBy() {
    StringBuilder sql = new StringBuilder();
    sql.append(getColumnName(CN_PRIORITY)).append(Backend.SQL_SORTASC);
    return sql.toString();
  }

  @Override
  public void getFields(ResultSetWrapper rs) {
    super.getFields(rs);
    if (rs.configureSection(CLASSVARIABLES)) {
      rs.configureColumn(CN_OBJECTCLASSNAME);
      rs.configureColumn(CN_OBJECTCLASSID);
      rs.configureColumn(CN_OBJECTID);
      rs.configureColumn(CN_DOMAINCONTEXTCLASSID);
      rs.configureColumn(CN_DOMAINCONTEXTID);
      rs.configureColumn(CN_GRANTEECLASSID);
      rs.configureColumn(CN_GRANTEEID);
      rs.configureColumn(CN_PRIORITY);
      rs.configureColumn(CN_PERMISSIONS);
      rs.configureColumn(CN_ALLOWED);
      rs.configureColumn(CN_MESSAGE);
      rs.configureColumn(CN_ID);
      rs.configureColumn(CN_SERIAL);
    }
    if (rs.getRow() <= 0) {
      throw new PersistenceException(getSession(), "no valid row");
    }
    objectClassName = rs.getString();
    objectClassId = rs.getInt();
    objectId = rs.getLong();
    domainContextClassId = rs.getInt();
    domainContextId = rs.getLong();
    granteeClassId = rs.getInt();
    granteeId = rs.getLong();
    priority = rs.getInt();
    permissions = rs.getString();
    allowed = rs.getBoolean();
    message = rs.getString();
    setId(rs.getLong());
    setSerial(rs.getLong());
  }

  @Override
  public int setFields(PreparedStatementWrapper st) {
    int ndx = super.setFields(st);
    st.setString(++ndx, objectClassName);
    st.setInt(++ndx, objectClassId);
    st.setLong(++ndx, objectId);
    st.setInt(++ndx, domainContextClassId);
    st.setLong(++ndx, domainContextId);
    st.setInt(++ndx, granteeClassId);
    st.setLong(++ndx, granteeId);
    st.setInt(++ndx, priority);
    st.setString(++ndx, permissions);
    st.setBoolean(++ndx, allowed);
    st.setString(++ndx, message);
    st.setLong(++ndx, getId());
    st.setLong(++ndx, getSerial());
    return ndx;
  }

  @Override
  public String createInsertSql() {
    return Backend.SQL_INSERT_INTO + getTableName() + Backend.SQL_LEFT_PARENTHESIS +
           CN_OBJECTCLASSNAME + Backend.SQL_COMMA +
           CN_OBJECTCLASSID + Backend.SQL_COMMA +
           CN_OBJECTID + Backend.SQL_COMMA +
           CN_DOMAINCONTEXTCLASSID + Backend.SQL_COMMA +
           CN_DOMAINCONTEXTID + Backend.SQL_COMMA +
           CN_GRANTEECLASSID + Backend.SQL_COMMA +
           CN_GRANTEEID + Backend.SQL_COMMA +
           CN_PRIORITY + Backend.SQL_COMMA +
           CN_PERMISSIONS + Backend.SQL_COMMA +
           CN_ALLOWED + Backend.SQL_COMMA +
           CN_MESSAGE + Backend.SQL_COMMA +
           CN_ID + Backend.SQL_COMMA +
           CN_SERIAL +
           Backend.SQL_INSERT_VALUES +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR + Backend.SQL_RIGHT_PARENTHESIS;
  }

  @Override
  public String createUpdateSql() {
    return Backend.SQL_UPDATE + getTableName() + Backend.SQL_SET +
           CN_OBJECTCLASSNAME + Backend.SQL_EQUAL_PAR_COMMA +
           CN_OBJECTCLASSID + Backend.SQL_EQUAL_PAR_COMMA +
           CN_OBJECTID + Backend.SQL_EQUAL_PAR_COMMA +
           CN_DOMAINCONTEXTCLASSID + Backend.SQL_EQUAL_PAR_COMMA +
           CN_DOMAINCONTEXTID + Backend.SQL_EQUAL_PAR_COMMA +
           CN_GRANTEECLASSID + Backend.SQL_EQUAL_PAR_COMMA +
           CN_GRANTEEID + Backend.SQL_EQUAL_PAR_COMMA +
           CN_PRIORITY + Backend.SQL_EQUAL_PAR_COMMA +
           CN_PERMISSIONS + Backend.SQL_EQUAL_PAR_COMMA +
           CN_ALLOWED + Backend.SQL_EQUAL_PAR_COMMA +
           CN_MESSAGE + Backend.SQL_EQUAL_PAR_COMMA +
           CN_SERIAL + Backend.SQL_EQUAL + CN_SERIAL + Backend.SQL_PLUS_ONE +
           Backend.SQL_WHERE + CN_ID + Backend.SQL_EQUAL_PAR +
           Backend.SQL_AND + CN_SERIAL + Backend.SQL_EQUAL_PAR;
  }

  @Override
  public String getObjectClassName()    {
    return objectClassName;
  }

  @Override
  public void setObjectClassName(String objectClassName) {
    if (!Objects.equals(this.objectClassName, objectClassName)) {
      setModified(true);
      this.objectClassName = objectClassName;
    }
  }

  @Override
  public int getObjectClassId()    {
    return objectClassId;
  }

  @Override
  public void setObjectClassId(int objectClassId) {
    if (this.objectClassId != objectClassId) {
      setModified(true);
      this.objectClassId = objectClassId;
    }
  }

  @Override
  public long getObjectId()    {
    return objectId;
  }

  @Override
  public void setObjectId(long objectId) {
    if (this.objectId != objectId) {
      setModified(true);
      this.objectId = objectId;
    }
  }

  @Override
  public int getDomainContextClassId()    {
    return domainContextClassId;
  }

  @Override
  public void setDomainContextClassId(int domainContextClassId) {
    if (this.domainContextClassId != domainContextClassId) {
      setModified(true);
      this.domainContextClassId = domainContextClassId;
    }
  }

  @Override
  public long getDomainContextId()    {
    return domainContextId;
  }

  @Override
  public void setDomainContextId(long domainContextId) {
    if (this.domainContextId != domainContextId) {
      setModified(true);
      this.domainContextId = domainContextId;
    }
  }

  @Override
  public int getGranteeClassId()    {
    return granteeClassId;
  }

  @Override
  public void setGranteeClassId(int granteeClassId) {
    if (this.granteeClassId != granteeClassId) {
      setModified(true);
      this.granteeClassId = granteeClassId;
    }
  }

  @Override
  public long getGranteeId()    {
    return granteeId;
  }

  @Override
  public void setGranteeId(long granteeId) {
    if (this.granteeId != granteeId) {
      setModified(true);
      this.granteeId = granteeId;
    }
  }

  @Override
  public int getPriority()    {
    return priority;
  }

  @Override
  public void setPriority(int priority) {
    if (this.priority != priority) {
      setModified(true);
      this.priority = priority;
    }
  }

  @Override
  public String getPermissions()    {
    return permissions;
  }

  @Override
  public void setPermissions(String permissions) {
    if (!Objects.equals(this.permissions, permissions)) {
      setModified(true);
      this.permissions = permissions;
    }
  }

  @Override
  public boolean isAllowed()    {
    return allowed;
  }

  @Override
  public void setAllowed(boolean allowed) {
    if (this.allowed != allowed) {
      setModified(true);
      this.allowed = allowed;
    }
  }

  @Override
  public String getMessage()    {
    return message;
  }

  @Override
  public void setMessage(String message) {
    if (!Objects.equals(this.message, message)) {
      setModified(true);
      this.message = message;
    }
  }

  /**
   * Copies all attributes from a snapshot back to this object.
   *
   * @param snapshot the snapshot object
   */
  protected void revertAttributesToSnapshot(SecurityPersistenceImpl snapshot) {
    super.revertAttributesToSnapshot(snapshot);
    objectClassName = snapshot.objectClassName;
    objectClassId = snapshot.objectClassId;
    objectId = snapshot.objectId;
    domainContextClassId = snapshot.domainContextClassId;
    domainContextId = snapshot.domainContextId;
    granteeClassId = snapshot.granteeClassId;
    granteeId = snapshot.granteeId;
    priority = snapshot.priority;
    permissions = snapshot.permissions;
    allowed = snapshot.allowed;
    message = snapshot.message;
  }

  //</editor-fold>//GEN-END:methods

}

