/*
 * Tentackle - https://tentackle.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package org.tentackle.persist.ns;

import org.tentackle.dbms.PreparedStatementWrapper;
import org.tentackle.dbms.ResultSetWrapper;
import org.tentackle.dbms.StatementId;
import org.tentackle.misc.IdentifiableMap;
import org.tentackle.misc.TrackedArrayList;
import org.tentackle.misc.TrackedList;
import org.tentackle.ns.pdo.NumberPool;
import org.tentackle.ns.pdo.NumberPoolPersistence;
import org.tentackle.ns.pdo.NumberRange;
import org.tentackle.pdo.DomainContext;
import org.tentackle.pdo.PersistentDomainObject;
import org.tentackle.pdo.PersistentObjectService;
import org.tentackle.persist.AbstractPersistentObject;
import org.tentackle.persist.PersistentObjectClassVariables;
import org.tentackle.persist.ns.rmi.NumberPoolRemoteDelegate;
import org.tentackle.session.PersistenceException;
import org.tentackle.session.Session;
import org.tentackle.sql.Backend;
import org.tentackle.validate.ValidationFailedException;
import org.tentackle.validate.ValidationResult;
import org.tentackle.validate.ValidationScope;
import org.tentackle.validate.ValidationUtilities;

import java.io.Serial;
import java.rmi.RemoteException;
import java.util.List;
import java.util.Objects;



/**
 * Number Space persistence implementation.
 *
 * @author harald
 */
@PersistentObjectService(NumberPool.class)
public class NumberPoolPersistenceImpl extends AbstractPersistentObject<NumberPool, NumberPoolPersistenceImpl> implements NumberPoolPersistence {

  @Serial
  private static final long serialVersionUID = 1L;


  // @wurblet classVariables ClassVariables

  //<editor-fold defaultstate="collapsed" desc="code 'classVariables' generated by wurblet ClassVariables">//GEN-BEGIN:classVariables

  /** Variables common to all instances of NumberPoolPersistenceImpl. */
  public static final PersistentObjectClassVariables<NumberPool, NumberPoolPersistenceImpl> CLASSVARIABLES =
            PersistentObjectClassVariables.create(
                    NumberPool.class,
                    NumberPoolPersistenceImpl.class,
                    "npool",
                    null,
                    null);

  @Override
  public PersistentObjectClassVariables<NumberPool, NumberPoolPersistenceImpl> getClassVariables() {
    return CLASSVARIABLES;
  }

  //</editor-fold>//GEN-END:classVariables

  // @wurblet fieldnames ColumnNames

  //<editor-fold defaultstate="collapsed" desc="code 'fieldnames' generated by wurblet ColumnNames">//GEN-BEGIN:fieldnames


  /** database column name for 'name'. */
  public static final String CN_NAME = "name";

  /** database column name for 'realm'. */
  public static final String CN_REALM = "realm";

  /** database column name for 'description'. */
  public static final String CN_DESCRIPTION = "description";

  /** database column name for 'online'. */
  public static final String CN_ONLINE = "poolonline";

  /** database column name for 'lowWaterMark'. */
  public static final String CN_LOWWATERMARK = "lowmark";

  /** database column name for 'requestSize'. */
  public static final String CN_REQUESTSIZE = "reqsize";

  /** database column name for 'uplink'. */
  public static final String CN_UPLINK = "uplink";

  //</editor-fold>//GEN-END:fieldnames

  // @wurblet declare Declare

  //<editor-fold defaultstate="collapsed" desc="code 'declare' generated by wurblet Declare">//GEN-BEGIN:declare


  /** the number pool name. */
  private String name;
  /** the last persisted value of name. */
  private String namePersisted;

  /** pool realm, optional. */
  private String realm;
  /** the last persisted value of realm. */
  private String realmPersisted;

  /** short description. */
  private String description;
  /** the last persisted value of description. */
  private String descriptionPersisted;

  /** true if online, else offline. */
  private boolean online;
  /** the last persisted value of online. */
  private boolean onlinePersisted;

  /** minimum number count before request to fill up from uplink, 0 if disable. */
  private long lowWaterMark;
  /** the last persisted value of lowWaterMark. */
  private long lowWaterMarkPersisted;

  /** number count to request from uplink, 0 if disable. */
  private long requestSize;
  /** the last persisted value of requestSize. */
  private long requestSizePersisted;

  /** uplink configuration (optional). */
  private String uplink;
  /** the last persisted value of uplink. */
  private String uplinkPersisted;

  //</editor-fold>//GEN-END:declare


  /**
   * Creates a number space.
   *
   * @param pdo the number space PDO
   * @param context the domain context
   */
  public NumberPoolPersistenceImpl (NumberPool pdo, DomainContext context)    {
    super(pdo, context);
  }

  /**
   * Creates a number space with a session only.
   *
   * @param pdo the number space PDO
   * @param session the session
   */
  public NumberPoolPersistenceImpl(NumberPool pdo, Session session) {
    super(pdo, session);
  }

  /**
   * Creates a number space without domain context or session.
   *
   * @param pdo the number space PDO
   */
  public NumberPoolPersistenceImpl(NumberPool pdo) {
    super(pdo);
  }

  /**
   * Creates a number space without domain context or session.
   */
  public NumberPoolPersistenceImpl() {
    super();
  }

  @Override
  public boolean isUpdatingSerialEvenIfNotModified() {
    // don't update the pool if only ranges changed (even though it's a root entity)
    return false;
  }

  // @wurblet methods MethodsImpl

  //<editor-fold defaultstate="collapsed" desc="code 'methods' generated by wurblet MethodsImpl">//GEN-BEGIN:methods


  @Override
  public NumberPoolRemoteDelegate getRemoteDelegate() {
    return (NumberPoolRemoteDelegate) super.getRemoteDelegate();
  }

  @Override
  public boolean isRootEntity() {
    return true;
  }

  @Override
  public boolean isTracked() {
    return true;    // invoking isModified() is ok
  }

  @Override
  public void setModified(boolean modified) {
    super.setModified(modified);
    if (!modified) {
      namePersisted = name;
      realmPersisted = realm;
      descriptionPersisted = description;
      onlinePersisted = online;
      lowWaterMarkPersisted = lowWaterMark;
      requestSizePersisted = requestSize;
      uplinkPersisted = uplink;
    }
  }

  @Override
  public boolean differsPersisted() {
    return super.differsPersisted()
           || !Objects.equals(namePersisted, name)
           || !Objects.equals(realmPersisted, realm)
           || !Objects.equals(descriptionPersisted, description)
           || onlinePersisted != online
           || lowWaterMarkPersisted != lowWaterMark
           || requestSizePersisted != requestSize
           || !Objects.equals(uplinkPersisted, uplink)
           ;
  }

  @Override
  public boolean attributesModified() {
    return isForcedModified() || differsPersisted();
  }

  @Override
  public void getFields(ResultSetWrapper rs) {
    super.getFields(rs);
    if (rs.configureSection(CLASSVARIABLES)) {
      rs.configureColumn(CN_NAME);
      rs.configureColumn(CN_REALM);
      rs.configureColumn(CN_DESCRIPTION);
      rs.configureColumn(CN_ONLINE);
      rs.configureColumn(CN_LOWWATERMARK);
      rs.configureColumn(CN_REQUESTSIZE);
      rs.configureColumn(CN_UPLINK);
      rs.configureColumn(CN_ID);
      rs.configureColumn(CN_SERIAL);
    }
    if (rs.getRow() <= 0) {
      throw new PersistenceException(getSession(), "no valid row");
    }
    name = rs.getString();
    realm = rs.getString(true);
    description = rs.getString();
    online = rs.getBoolean();
    lowWaterMark = rs.getLong();
    requestSize = rs.getLong();
    uplink = rs.getString();
    setId(rs.getLong());
    setSerial(rs.getLong());
  }

  @Override
  public int setFields(PreparedStatementWrapper st) {
    int ndx = super.setFields(st);
    st.setString(++ndx, name);
    st.setString(++ndx, realm, true);
    st.setString(++ndx, description);
    st.setBoolean(++ndx, online);
    st.setLong(++ndx, lowWaterMark);
    st.setLong(++ndx, requestSize);
    st.setString(++ndx, uplink);
    st.setLong(++ndx, getId());
    st.setLong(++ndx, getSerial());
    return ndx;
  }

  @Override
  public String createInsertSql() {
    return Backend.SQL_INSERT_INTO + getTableName() + Backend.SQL_LEFT_PARENTHESIS +
           CN_NAME + Backend.SQL_COMMA +
           CN_REALM + Backend.SQL_COMMA +
           CN_DESCRIPTION + Backend.SQL_COMMA +
           CN_ONLINE + Backend.SQL_COMMA +
           CN_LOWWATERMARK + Backend.SQL_COMMA +
           CN_REQUESTSIZE + Backend.SQL_COMMA +
           CN_UPLINK + Backend.SQL_COMMA +
           CN_ID + Backend.SQL_COMMA +
           CN_SERIAL +
           Backend.SQL_INSERT_VALUES +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR_COMMA +
           Backend.SQL_PAR + Backend.SQL_RIGHT_PARENTHESIS;
  }

  @Override
  public String createUpdateSql() {
    return Backend.SQL_UPDATE + getTableName() + Backend.SQL_SET +
           CN_NAME + Backend.SQL_EQUAL_PAR_COMMA +
           CN_REALM + Backend.SQL_EQUAL_PAR_COMMA +
           CN_DESCRIPTION + Backend.SQL_EQUAL_PAR_COMMA +
           CN_ONLINE + Backend.SQL_EQUAL_PAR_COMMA +
           CN_LOWWATERMARK + Backend.SQL_EQUAL_PAR_COMMA +
           CN_REQUESTSIZE + Backend.SQL_EQUAL_PAR_COMMA +
           CN_UPLINK + Backend.SQL_EQUAL_PAR_COMMA +
           CN_SERIAL + Backend.SQL_EQUAL + CN_SERIAL + Backend.SQL_PLUS_ONE +
           Backend.SQL_WHERE + CN_ID + Backend.SQL_EQUAL_PAR +
           Backend.SQL_AND + CN_SERIAL + Backend.SQL_EQUAL_PAR;
  }

  @Override
  public String getName()    {
    return name;
  }

  @Override
  public void setName(String name) {
    if (!Objects.equals(this.name, name)) {
      assertMutable();
      firePropertyChange(AN_NAME, this.name, name);
      this.name = name;
    }
  }

  @Override
  public boolean isNameModified() {
    return !Objects.equals(name, namePersisted);
  }

  @Override
  public String getNamePersisted() {
    return namePersisted;
  }

  @Override
  public String getRealm()    {
    return realm;
  }

  @Override
  public void setRealm(String realm) {
    if (!Objects.equals(this.realm, realm)) {
      assertMutable();
      firePropertyChange(AN_REALM, this.realm, realm);
      this.realm = realm;
    }
  }

  @Override
  public boolean isRealmModified() {
    return !Objects.equals(realm, realmPersisted);
  }

  @Override
  public String getRealmPersisted() {
    return realmPersisted;
  }

  @Override
  public String getDescription()    {
    return description;
  }

  @Override
  public void setDescription(String description) {
    if (!Objects.equals(this.description, description)) {
      assertMutable();
      firePropertyChange(AN_DESCRIPTION, this.description, description);
      this.description = description;
    }
  }

  @Override
  public boolean isDescriptionModified() {
    return !Objects.equals(description, descriptionPersisted);
  }

  @Override
  public String getDescriptionPersisted() {
    return descriptionPersisted;
  }

  @Override
  public boolean isOnline()    {
    return online;
  }

  @Override
  public void setOnline(boolean online) {
    if (this.online != online) {
      assertMutable();
      firePropertyChange(AN_ONLINE, this.online, online);
      this.online = online;
    }
  }

  @Override
  public boolean isOnlineModified() {
    return online != onlinePersisted;
  }

  @Override
  public boolean isOnlinePersisted() {
    return onlinePersisted;
  }

  @Override
  public long getLowWaterMark()    {
    return lowWaterMark;
  }

  @Override
  public void setLowWaterMark(long lowWaterMark) {
    if (this.lowWaterMark != lowWaterMark) {
      assertMutable();
      firePropertyChange(AN_LOWWATERMARK, this.lowWaterMark, lowWaterMark);
      this.lowWaterMark = lowWaterMark;
    }
  }

  @Override
  public boolean isLowWaterMarkModified() {
    return lowWaterMark != lowWaterMarkPersisted;
  }

  @Override
  public long getLowWaterMarkPersisted() {
    return lowWaterMarkPersisted;
  }

  @Override
  public long getRequestSize()    {
    return requestSize;
  }

  @Override
  public void setRequestSize(long requestSize) {
    if (this.requestSize != requestSize) {
      assertMutable();
      firePropertyChange(AN_REQUESTSIZE, this.requestSize, requestSize);
      this.requestSize = requestSize;
    }
  }

  @Override
  public boolean isRequestSizeModified() {
    return requestSize != requestSizePersisted;
  }

  @Override
  public long getRequestSizePersisted() {
    return requestSizePersisted;
  }

  @Override
  public String getUplink()    {
    return uplink;
  }

  @Override
  public void setUplink(String uplink) {
    if (!Objects.equals(this.uplink, uplink)) {
      assertMutable();
      firePropertyChange(AN_UPLINK, this.uplink, uplink);
      this.uplink = uplink;
    }
  }

  @Override
  public boolean isUplinkModified() {
    return !Objects.equals(uplink, uplinkPersisted);
  }

  @Override
  public String getUplinkPersisted() {
    return uplinkPersisted;
  }

  /**
   * Copies all attributes from a snapshot back to this object.
   *
   * @param snapshot the snapshot object
   */
  protected void revertAttributesToSnapshot(NumberPoolPersistenceImpl snapshot) {
    super.revertAttributesToSnapshot(snapshot);
    name = snapshot.name;
    namePersisted = snapshot.namePersisted;
    realm = snapshot.realm;
    realmPersisted = snapshot.realmPersisted;
    description = snapshot.description;
    descriptionPersisted = snapshot.descriptionPersisted;
    online = snapshot.online;
    onlinePersisted = snapshot.onlinePersisted;
    lowWaterMark = snapshot.lowWaterMark;
    lowWaterMarkPersisted = snapshot.lowWaterMarkPersisted;
    requestSize = snapshot.requestSize;
    requestSizePersisted = snapshot.requestSizePersisted;
    uplink = snapshot.uplink;
    uplinkPersisted = snapshot.uplinkPersisted;
  }

  // selects by unique domain key
  // @wurblet selectByUniqueDomainKey PdoSelectUnique name

  //</editor-fold>//GEN-END:methods

  //<editor-fold defaultstate="collapsed" desc="code 'selectByUniqueDomainKey' generated by wurblet PdoSelectUnique/MethodsImpl">//GEN-BEGIN:selectByUniqueDomainKey


  private static final StatementId SELECT_BY_UNIQUE_DOMAIN_KEY_STMT = new StatementId();

  @Override
  public NumberPool selectByUniqueDomainKey(String name) {
    if (getSession().isRemote())  {
      try {
        NumberPool obj = getRemoteDelegate().selectByUniqueDomainKey(getDomainContext(), name);
        configureRemoteObject(getDomainContext(), obj);
        return obj;
      }
      catch (RemoteException e) {
        throw PersistenceException.createFromRemoteException(this, e);
      }
    }
    PreparedStatementWrapper st = getPreparedStatement(SELECT_BY_UNIQUE_DOMAIN_KEY_STMT,
      () -> {
        StringBuilder sql = createSelectAllInnerSql();
        sql.append(Backend.SQL_AND);
        sql.append(getColumnName(CN_NAME));
        sql.append(Backend.SQL_EQUAL_PAR);
        getBackend().buildSelectSql(sql, false, 0, 0);
        return sql.toString();
      }
    );
    int ndx = 1;
    st.setString(ndx, name);
    return executeFirstPdoQuery(st);
  }

  //</editor-fold>//GEN-END:selectByUniqueDomainKey

  // @wurblet relations PdoRelations

  //<editor-fold defaultstate="collapsed" desc="code 'relations' generated by wurblet PdoRelations">//GEN-BEGIN:relations


  // composite list of NumberRange numberRangeList via NumberRange#numberPoolId (NumberRange)
  private TrackedList<NumberRange> numberRangeList;
  private boolean numberRangeListLoaded;
  private transient TrackedList<NumberRange> numberRangeListSnapshot;

  @Override
  public TrackedList<NumberRange> getNumberRangeList()  {
    if (!numberRangeListLoaded) {
      numberRangeList = isNew() ? new TrackedArrayList<>(false) : on(NumberRange.class).selectByNumberPoolId(getId());
      for (NumberRange obj: numberRangeList)  {
        obj.setNumberPool(me());
      }
      if (isImmutable()) {
        numberRangeList.setImmutable(true);
      }
      numberRangeListLoaded = true;
    }
    return numberRangeList;
  }

  /**
   * Gets numberRangeList without performing a select if not loaded.
   *
   * @return numberRangeList NumberRange
   */
  public TrackedList<NumberRange> getNumberRangeListBlunt() {
    if (!numberRangeListLoaded) {
      numberRangeList = new TrackedArrayList<>(false);
      if (isImmutable()) {
        numberRangeList.setImmutable(true);
      }
      numberRangeListLoaded = true;
    }
    return numberRangeList;
  }

  @Override
  public boolean isNumberRangeListLoaded() {
    return numberRangeListLoaded;
  }

  @Override
  public void setSession(Session session)  {
    super.setSession(session);
    session.applyTo(numberRangeList);
  }

  @Override
  public void setDomainContext(DomainContext context)  {
    super.setDomainContext(context);
    context.applyTo(numberRangeList);
  }

  @Override
  public void deleteReferencingRelations() {
    super.deleteReferencingRelations();
    deleteReferencingRelations(true);
  }


  /**
   * Deletes referencing composite relations.
   * This method is invoked from deleteReferencingRelations() or saveReferencingRelations().
   * When invoked from saveReferencingRelations() lazy relations not
   * already loaded will be considered to be unchanged, i.e.
   * need no update or insert.
   *
   * @param delete is true if invoked from deleteReferencingRelations()
   */
  private void deleteReferencingRelations(boolean delete)  {
    if (delete) {
      delete(getNumberRangeList());
    }
    if (numberRangeList != null && numberRangeList.isSomeRemoved()) {
      delete(numberRangeList.getRemovedObjects());
    }
  }

  @Override
  public void saveReferencingRelations(boolean update)  {
    super.saveReferencingRelations(update);
    if (update) {
      deleteReferencingRelations(false);
    }
    if (numberRangeList != null) {
      getDomainContext().applyTo(numberRangeList);
      for (NumberRange obj: numberRangeList)  {
        obj.setNumberPool(me());
      }
      save(numberRangeList, true);
    }
  }

  @Override
  public List<ValidationResult> validate(String validationPath, ValidationScope scope) {
    List<ValidationResult> results = super.validate(validationPath, scope);
    try {
      if (numberRangeList != null) {
        results.addAll(ValidationUtilities.getInstance().validateCollection(
                  numberRangeList, validationPath + ".numberRangeList", scope));
      }
    }
    catch (ValidationFailedException vfx) {
      vfx.reThrow(results);
    }
    return results;
  }

  @Override
  public void setImmutable(boolean immutable) {
    super.setImmutable(immutable);
    if (numberRangeList != null) {
      numberRangeList.setImmutable(immutable);
    }
  }

  @Override
  public boolean isModified()  {
    return super.isModified()
           || isModified(numberRangeList)
           ;
  }

  @Override
  public boolean isComposite()  {
    return true;
  }


  @Override
  public IdentifiableMap<? extends PersistentDomainObject<?>> loadComponents(boolean onlyLoaded) {
    IdentifiableMap<PersistentDomainObject<?>> components = new IdentifiableMap<>();
    addComponents(components, onlyLoaded);
    return components;
  }

  @Override
  public int addComponents(IdentifiableMap<PersistentDomainObject<?>> components, boolean onlyLoaded)  {
    int count = 0;
    count += super.addComponents(components, onlyLoaded);
    if (!onlyLoaded || numberRangeListLoaded) {
      count += addComponents(components, getNumberRangeList(), onlyLoaded);
    }
    return count;
  }

  @Override
  public void insertPlainWithComponents()  {
    insertPlain();
    insertPlainWithComponents(getNumberRangeList());
  }

  @Override
  public void deletePlainWithComponents()  {
    deletePlainWithComponents(getNumberRangeList());
    deletePlain();
  }

  @Override
  public void markDeleted() {
    super.markDeleted();
    markDeleted(numberRangeList);
  }

  /**
   * Updates the components in snapshot object.<br>
   * The snapshot object is assumed to be a clone of this object.
   *
   * @param snapshot the snapshot
   */
  protected void createComponentsInSnapshot(NumberPoolPersistenceImpl snapshot) {
    super.createComponentsInSnapshot(snapshot);
    snapshot.numberRangeListSnapshot = TrackedList.createSnapshot(numberRangeList);
  }

  /**
   * Reverts all components of this object to a given snapshot.
   *
   * @param snapshot the snapshot object
   */
  protected void revertComponentsToSnapshot(NumberPoolPersistenceImpl snapshot) {
    super.revertComponentsToSnapshot(snapshot);
    numberRangeList = TrackedList.revertToSnapshot(numberRangeList, snapshot.numberRangeListSnapshot);
    numberRangeListLoaded = snapshot.numberRangeListLoaded;
  }

  //</editor-fold>//GEN-END:relations

  // @wurblet selectByNameRealm PdoSelectUnique name realm

  //<editor-fold defaultstate="collapsed" desc="code 'selectByNameRealm' generated by wurblet PdoSelectUnique">//GEN-BEGIN:selectByNameRealm


  private static final StatementId SELECT_BY_NAME_REALM_STMT = new StatementId();

  @Override
  public NumberPool selectByNameRealm(String name, String realm) {
    if (getSession().isRemote())  {
      try {
        NumberPool obj = getRemoteDelegate().selectByNameRealm(getDomainContext(), name, realm);
        configureRemoteObject(getDomainContext(), obj);
        return obj;
      }
      catch (RemoteException e) {
        throw PersistenceException.createFromRemoteException(this, e);
      }
    }
    PreparedStatementWrapper st = getPreparedStatement(SELECT_BY_NAME_REALM_STMT,
      () -> {
        StringBuilder sql = createSelectAllInnerSql();
        sql.append(Backend.SQL_AND);
        sql.append(getColumnName(CN_NAME));
        sql.append(Backend.SQL_EQUAL_PAR);
        sql.append(Backend.SQL_AND);
        sql.append(getColumnName(CN_REALM));
        sql.append(Backend.SQL_EQUAL_PAR);
        getBackend().buildSelectSql(sql, false, 0, 0);
        return sql.toString();
      }
    );
    int ndx = 1;
    st.setString(ndx++, name);
    st.setString(ndx, realm, true);
    return executeFirstPdoQuery(st);
  }

  //</editor-fold>//GEN-END:selectByNameRealm

  @Override
  public void reloadRanges() {
    numberRangeListLoaded = false;
    numberRangeList = null;
  }

}

