package org.t3as.snomedct.lookup.cmdline;

/*
 * #%L
 * NICTA t3as UMLS CUI to SNOMED CT Lookup
 * %%
 * Copyright (C) 2014 NICTA
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * 
 * Additional permission under GNU GPL version 3 section 7
 * 
 * If you modify this Program, or any covered work, by linking or combining
 * it with H2, GWT, or JavaBeans Activation Framework (JAF) (or a
 * modified version of those libraries), containing parts covered by the
 * terms of the H2 License, the GWT Terms, or the Common Development and
 * Distribution License (CDDL) version 1.0 ,the licensors of this Program
 * grant you additional permission to convey the resulting work.
 * #L%
 */

import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import org.t3as.metamap.JaxbLoader;
import org.t3as.metamap.jaxb.MMOs;
import org.t3as.snomedct.lookup.SnomedLookup;
import org.t3as.snomedct.lookup.SnomedTerm;
import org.xml.sax.SAXException;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.parsers.ParserConfigurationException;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.sql.SQLException;

/**
 * Reads the --XMLf1 output generated by the metamap-tagger tool and does lookups for the corresponding SNOMED CT
 * concepts in UMLS, enriching and printing the XML out.
 */
public class Main {

    public static void main(final String[] args)
            throws IOException, SQLException, JAXBException, ParserConfigurationException, SAXException {
        final Options opts = new Options();
        JCommander jc = null;
        try { jc = new JCommander(opts, args);
        } catch (final Exception e) {
            System.err.println("Could not parse the options: " + e.getMessage());
            System.exit(1);
        }
        if (opts.showUsage) {
            jc.usage();
            System.exit(0);
        }
        if (opts.infile == null && !(opts.cuiDesc || opts.snomedIds)) {
            System.out.println("If not specifying -infile, please specify either -cuiDesc or -snomedId.");
            jc.usage();
            System.exit(1);
        }

        // if there is no file to read from, read CUI's and descriptions from stdin
        if (opts.infile == null) {
            try (final BufferedReader reader = new BufferedReader(new InputStreamReader(System.in))) {
                try (final SnomedLookup snomedLookup = new SnomedLookup(opts.dbFile)) {
                    for (String line; (line = reader.readLine()) != null; ) {
                        SnomedTerm term = null;
                        if (opts.cuiDesc) {
                            final String[] parts = line.split(" ", 2);
                            term = snomedLookup.findFromCuiAndDesc(parts[0], parts[1]);
                        }
                        else if(opts.snomedIds) {
                            term = snomedLookup.findFromSnomedId(line);
                        }
                        if (term != null) {
                            System.out.println(term);
                        }
                    }
                }
            }
        }
        else {
            // look up the SNOMED codes from the UMLS CUI codes returned by MetaMap
            try (final SnomedLookup snomedLookup = new SnomedLookup(opts.dbFile)) {
                final long jaxbStart = System.currentTimeMillis();
                final MMOs root = JaxbLoader.loadXml(opts.infile);
                final long jaxbEnd = System.currentTimeMillis();

                final long start = System.currentTimeMillis();
                final int count = snomedLookup.enrichXml(root);
                final long end = System.currentTimeMillis();

                // print the enriched XML to stdout
                final JAXBContext jaxbContext = JAXBContext.newInstance(MMOs.class);
                final Marshaller m = jaxbContext.createMarshaller();
                m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
                m.marshal(root, System.out);

                System.err.printf("Loaded XML in %,d millis, looked up %,d SNOMED concepts in %,d milliseconds.\n",
                        jaxbEnd - jaxbStart, count, end - start);
            }
        }
    }

    private static class Options {
        @Parameter(help = true, names = {"-h", "--help"}, description = "Show this help message.")
        boolean showUsage = false;

        @Parameter(names = "-infile", description = "MetaMap '--XMLf1' output to look up SNOMED CT codes for.")
        File infile = null;

        @Parameter(names = "-cuiDescs", description = "Read CUI and descriptions to lookup from stdin.")
        boolean cuiDesc = false;

        @Parameter(names = "-snomedIds", description = "Read SNOMED CT ID's to lookup from stdin.")
        boolean snomedIds = false;

        @Parameter(names = "-dbFile", description = "The file prefix to the SNOMED CT lookup database.")
        File dbFile = new File("/opt/snomed-coder-web/data/snomedct");
    }
}
