package org.symphonyoss.symphony.apps.authentication;

import org.apache.commons.codec.binary.Hex;
import org.symphonyoss.symphony.apps.authentication.tokens.StoreTokensProvider;
import org.symphonyoss.symphony.apps.authentication.tokens.StoreTokensProviderFactory;
import org.symphonyoss.symphony.apps.authentication.tokens.model.AppToken;

import java.security.SecureRandom;

/**
 * Service class to perform app authentication.
 *
 * Created by rsanchez on 06/03/18.
 */
public class AppAuthenticationService {

  private SecureRandom secureRandom = new SecureRandom();

  private AppAuthenticationApiClient apiClient = new AppAuthenticationApiClient();

  private StoreTokensProviderFactory factory = StoreTokensProviderFactory.getInstance();

  /**
   * Start the JWT authentication between the App and the POD.
   *
   * @param appId Application identifier.
   * @return The generated Application Token (Ta).
   * @throws AppAuthenticationException Failure to authenticate application
   */
  public AppToken authenticate(String appId) throws AppAuthenticationException {
    String appToken = generateToken();

    AppToken result = apiClient.authenticate(appId, appToken);

    StoreTokensProvider provider = factory.getComponent();
    provider.saveAppAuthenticationToken(result);

    return result;
  }

  /**
   * Generate a 64-bit security-safe String random token.
   * @return  String random token.
   */
  private String generateToken() {
    byte[] randBytes = new byte[64];
    secureRandom.nextBytes(randBytes);
    return Hex.encodeHexString(randBytes);
  }

  /**
   * Validate if the Symphony previously generated token by the app token and the POD token are
   * valid.
   *
   * @param applicationToken App token generated by the application.
   * @param symphonyToken Symphony token generated by the POD.
   * @return <code>true</code> if the token pair is valid.
   */
  public boolean isValidTokenPair(String applicationToken, String symphonyToken) {
    StoreTokensProvider provider = factory.getComponent();

    AppToken token = provider.getAppAuthenticationToken(applicationToken);

    if (token == null) {
      return false;
    }

    return symphonyToken.equals(token.getSymphonyToken());
  }

}
