// This file is part of JavaSMT,
// an API wrapper for a collection of SMT solvers:
// https://github.com/sosy-lab/java-smt
//
// SPDX-FileCopyrightText: 2024 Dirk Beyer <https://www.sosy-lab.org>
//
// SPDX-License-Identifier: Unlicense OR Apache-2.0 OR MIT

package org.sosy_lab.java_smt.example;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.logging.Level;
import org.sosy_lab.common.ShutdownNotifier;
import org.sosy_lab.common.configuration.Configuration;
import org.sosy_lab.common.configuration.InvalidConfigurationException;
import org.sosy_lab.common.log.BasicLogManager;
import org.sosy_lab.common.log.LogManager;
import org.sosy_lab.java_smt.SolverContextFactory;
import org.sosy_lab.java_smt.SolverContextFactory.Solvers;
import org.sosy_lab.java_smt.api.BooleanFormula;
import org.sosy_lab.java_smt.api.BooleanFormulaManager;
import org.sosy_lab.java_smt.api.IntegerFormulaManager;
import org.sosy_lab.java_smt.api.Model;
import org.sosy_lab.java_smt.api.NumeralFormula.IntegerFormula;
import org.sosy_lab.java_smt.api.ProverEnvironment;
import org.sosy_lab.java_smt.api.SolverContext;
import org.sosy_lab.java_smt.api.SolverContext.ProverOptions;
import org.sosy_lab.java_smt.api.SolverException;

/**
 * Compute the content of two arrays of size N and N-1, such that: - each item [i] from the second
 * array is the sum of the items [i] and [i+1] of the first array. - all items are distinct.
 *
 * <p>The simple case uses arrays of size 5 and 4 and the numbers 1-9.
 *
 * <p>The program computes all solutions, except symmetrical cases.
 */
public class ArraySum {

  public static final int SIZE = 11; // overall size (2N+1), odd number required!

  // also works with other numbers like 3, 5, 7, 9, 11, 13, 15,...

  public static void main(String... args)
      throws InvalidConfigurationException, InterruptedException {
    Configuration config = Configuration.defaultConfiguration();
    LogManager logger = BasicLogManager.create(config);
    ShutdownNotifier notifier = ShutdownNotifier.createDummy();

    {
      Solvers solver = Solvers.Z3;
      try (SolverContext context =
              SolverContextFactory.createSolverContext(config, logger, notifier, solver);
          ProverEnvironment prover = context.newProverEnvironment(ProverOptions.GENERATE_MODELS)) {

        long start = System.currentTimeMillis();
        ArraySolver arraySolver = new ArraySolver(context, prover);

        int i = 0;
        while (arraySolver.hasNext()) {
          System.out.println("#" + ++i + ":");
          System.out.println(arraySolver.next());
        }

        long end = System.currentTimeMillis();
        System.out.println("    Time to solve: " + (end - start) + " ms");
      } catch (InvalidConfigurationException | UnsatisfiedLinkError e) {

        // on some machines we support only some solvers,
        // thus we can ignore these errors.
        logger.logUserException(Level.INFO, e, "Solver " + solver + " is not available.");

      } catch (UnsupportedOperationException e) {
        logger.logUserException(Level.INFO, e, e.getMessage());
      }
    }
  }

  private ArraySum() {}

  public static class ArraySolver implements Iterator<String> {

    final ProverEnvironment prover;
    final BooleanFormulaManager bmgr;
    final IntegerFormulaManager imgr;
    final IntegerFormula[] symbols;

    private ArraySolver(SolverContext pContext, ProverEnvironment pProver)
        throws InterruptedException {
      prover = pProver;
      bmgr = pContext.getFormulaManager().getBooleanFormulaManager();
      imgr = pContext.getFormulaManager().getIntegerFormulaManager();

      symbols = getSymbols();
      prover.push(bmgr.and(getRules()));
      prover.push(bmgr.and(getConnections()));
      prover.push(imgr.lessThan(symbols[0], symbols[SIZE - 1])); // avoid symmetrical solutions
    }

    IntegerFormula[] getSymbols() {
      final IntegerFormula[] newSymbols = new IntegerFormula[SIZE];
      for (int i = 0; i < SIZE; i++) {
        newSymbols[i] = imgr.makeVariable("x_" + i);
      }
      return newSymbols;
    }

    List<BooleanFormula> getRules() {
      final List<BooleanFormula> rules = new ArrayList<>();

      // each symbol has a value from 1 to SIZE
      IntegerFormula one = imgr.makeNumber(1);
      IntegerFormula size = imgr.makeNumber(SIZE);
      for (int i = 0; i < SIZE; i++) {
        rules.add(imgr.lessOrEquals(one, symbols[i]));
        rules.add(imgr.lessOrEquals(symbols[i], size));
      }

      // distinct numbers
      rules.add(imgr.distinct(List.of(symbols)));

      return rules;
    }

    List<BooleanFormula> getConnections() {
      final List<BooleanFormula> connections = new ArrayList<>();
      for (int i = 1; i < SIZE; i += 2) {
        connections.add(imgr.equal(symbols[i], imgr.add(symbols[i - 1], symbols[i + 1])));
      }
      return connections;
    }

    Integer[] getValues() throws SolverException {
      Integer[] solution = new Integer[SIZE];
      try (Model model = prover.getModel()) {
        for (int i = 0; i < SIZE; i++) {
          solution[i] = Objects.requireNonNull(model.evaluate(symbols[i])).intValue();
        }
      }
      return solution;
    }

    @Override
    public boolean hasNext() {
      try {
        boolean isUnsolvable = prover.isUnsat(); // the hard part
        return !isUnsolvable;
      } catch (SolverException | InterruptedException pE) {
        return false;
      }
    }

    @Override
    public String next() {
      try {
        boolean isUnsolvable = prover.isUnsat(); // the hard part
        Preconditions.checkState(!isUnsolvable);

        Integer[] solution = getValues();

        // add constraints for next model
        List<BooleanFormula> modelAssignments = new ArrayList<>();
        for (int i = 0; i < SIZE; i++) {
          modelAssignments.add(imgr.equal(symbols[i], imgr.makeNumber(solution[i])));
        }
        prover.push(bmgr.not(bmgr.and(modelAssignments)));

        return toString(solution);
      } catch (SolverException | InterruptedException pE) {
        return null;
      }
    }

    private String toString(Integer[] solution) {
      StringBuilder str = new StringBuilder();
      for (int i = 0; i < SIZE; i += 2) {
        str.append(Strings.padEnd(solution[i].toString(), 6, ' '));
      }
      str.append('\n');
      for (int i = 1; i < SIZE; i += 2) {
        str.append("   ").append(Strings.padEnd(solution[i].toString(), 3, ' '));
      }
      str.append('\n');
      return str.toString();
    }
  }
}
