package org.sosy_lab.java_smt.example;

import com.google.common.collect.Iterables;
import java.math.BigInteger;
import java.util.logging.Level;
import org.sosy_lab.common.ShutdownNotifier;
import org.sosy_lab.common.configuration.Configuration;
import org.sosy_lab.common.configuration.InvalidConfigurationException;
import org.sosy_lab.common.log.BasicLogManager;
import org.sosy_lab.common.log.LogManager;
import org.sosy_lab.java_smt.SolverContextFactory;
import org.sosy_lab.java_smt.SolverContextFactory.Solvers;
import org.sosy_lab.java_smt.api.BooleanFormula;
import org.sosy_lab.java_smt.api.BooleanFormulaManager;
import org.sosy_lab.java_smt.api.IntegerFormulaManager;
import org.sosy_lab.java_smt.api.Model.ValueAssignment;
import org.sosy_lab.java_smt.api.NumeralFormula.IntegerFormula;
import org.sosy_lab.java_smt.api.OptimizationProverEnvironment;
import org.sosy_lab.java_smt.api.OptimizationProverEnvironment.OptStatus;
import org.sosy_lab.java_smt.api.SolverContext;
import org.sosy_lab.java_smt.api.SolverContext.ProverOptions;
import org.sosy_lab.java_smt.api.SolverException;

/** Example for getting values from optimization. */
public class MinimizeTemp {

  public static void main(String... args)
      throws InvalidConfigurationException, SolverException, InterruptedException {
    Configuration config = Configuration.defaultConfiguration();
    LogManager logger = BasicLogManager.create(config);
    ShutdownNotifier notifier = ShutdownNotifier.createDummy();

    // Z3 supports optimization by default.
    // OptiMathSat can also be used (after being installed separately).
    Solvers solver = Solvers.Z3;

    // create solver context
    try (SolverContext context =
            SolverContextFactory.createSolverContext(config, logger, notifier, solver);
        OptimizationProverEnvironment prover =
            context.newOptimizationProverEnvironment(ProverOptions.GENERATE_MODELS)) {
      BooleanFormulaManager bfmgr = context.getFormulaManager().getBooleanFormulaManager();
      IntegerFormulaManager ifmgr = context.getFormulaManager().getIntegerFormulaManager();

      BooleanFormula rule = computeRule(bfmgr, ifmgr, prover);
      logger.log(Level.INFO, rule);
    }
  }

  private static BooleanFormula computeRule(
      BooleanFormulaManager bfmgr,
      IntegerFormulaManager ifmgr,
      OptimizationProverEnvironment prover)
      throws InterruptedException, SolverException {

    // create some variables and numerals
    IntegerFormula acThermostat = ifmgr.makeVariable("acThermostat");
    IntegerFormula num25 = ifmgr.makeNumber(25);
    IntegerFormula temp = ifmgr.makeVariable("temp");
    IntegerFormula num20 = ifmgr.makeNumber(20);
    BooleanFormula ac = bfmgr.makeVariable("ac");

    // Constraint:  acThermostat > 25
    prover.addConstraint(ifmgr.greaterThan(acThermostat, num25));
    @SuppressWarnings("unused")
    int handle = prover.minimize(acThermostat);
    OptStatus status = prover.check();
    assert status == OptStatus.OPT;

    // We have several possibilities to get a model value for a variable:
    // 1) Rational minimumValue = prover.lower(handle, Rational.ZERO).get();
    // 2) BigInteger minimumValue2 = prover.getModel().evaluate(acThermostat);
    // 3) receive a formula for the concrete value from the model.
    // We use the third approach.
    // TODO improve model interface, such that the iteration and cast are no longer needed ;-)
    ValueAssignment va =
        Iterables.tryFind(prover.getModel(), v -> v.getKey().equals(acThermostat)).get();
    IntegerFormula minValue = (IntegerFormula) va.getValueAsFormula();

    // do we need the next line? It depends on the environment of Hamada.
    minValue = ifmgr.subtract(minValue, ifmgr.makeNumber(BigInteger.ONE));

    // Rule: IF Temp > 20 THEN (AC and AC_Thermostat = minValue)
    BooleanFormula rule =
        bfmgr.implication(
            ifmgr.greaterThan(temp, num20), bfmgr.and(ac, ifmgr.equal(acThermostat, minValue)));
    return rule;
  }
}
