/*
 *  JavaSMT is an API wrapper for a collection of SMT solvers.
 *  This file is part of JavaSMT.
 *
 *  Copyright (C) 2007-2018  Dirk Beyer
 *  All rights reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.sosy_lab.java_smt.api.visitors;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.sosy_lab.java_smt.api.BooleanFormula;
import org.sosy_lab.java_smt.api.Formula;
import org.sosy_lab.java_smt.api.FormulaManager;
import org.sosy_lab.java_smt.api.FunctionDeclaration;
import org.sosy_lab.java_smt.api.QuantifiedFormulaManager.Quantifier;

/**
 * Utility class for dumping a formula in dot-format. This class can be used to see shared subtrees
 * and an overall structure of a formula.
 *
 * <p>Use in combination with {@link FormulaManager#visitRecursively} to avoid visiting parts twice.
 * The graph will then contains shared subtrees only once.
 */
public class FormulaToDotVisitor implements FormulaVisitor<TraversalProcess> {

  private int nextId = 0;
  private final StringBuilder str = new StringBuilder();
  private final Map<Formula, Integer> ids = new HashMap<>();

  @Override
  public TraversalProcess visitFreeVariable(Formula pF, String pName) {
    str.append(id(pF) + " [label=\"var:" + pName + "\"];\n");
    return TraversalProcess.CONTINUE;
  }

  @Override
  public TraversalProcess visitBoundVariable(Formula pF, int pDeBruijnIdx) {
    str.append(id(pF) + " [label=\"idx:" + pDeBruijnIdx + "\"];\n");
    return TraversalProcess.CONTINUE;
  }

  @Override
  public TraversalProcess visitConstant(Formula pF, Object pValue) {
    str.append(id(pF) + " [label=\"const:" + pValue + "\"];\n");
    return TraversalProcess.CONTINUE;
  }

  @Override
  public TraversalProcess visitFunction(
      Formula pF, List<Formula> pArgs, FunctionDeclaration<?> pFunctionDeclaration) {
    int id = id(pF);
    str.append(id + " [label=\"" + pFunctionDeclaration.getKind() + "\"];\n");
    for (Formula var : pArgs) {
      str.append(id + " -> " + id(var) + ";\n");
    }
    return TraversalProcess.CONTINUE;
  }

  @Override
  public TraversalProcess visitQuantifier(
      BooleanFormula pF,
      Quantifier pQuantifier,
      List<Formula> pBoundVariables,
      BooleanFormula pBody) {
    int id = id(pF);
    str.append(id + "[label=\"" + pQuantifier + "\"];\n");
    for (Formula var : pBoundVariables) {
      str.append(id + " -> " + id(var) + ";\n");
    }
    str.append(id + " -> " + id(pBody) + ";\n");
    return TraversalProcess.CONTINUE;
  }

  private int id(Formula f) {
    Integer id = ids.get(f);
    if (id == null) {
      id = nextId++;
      ids.put(f, id);
    }
    return id;
  }

  @Override
  public String toString() {
    return "digraph FORMULA {\n" + str + "}";
  }
}
