package org.sosy_lab.java_smt.example;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import org.sosy_lab.common.ShutdownNotifier;
import org.sosy_lab.common.configuration.Configuration;
import org.sosy_lab.common.configuration.InvalidConfigurationException;
import org.sosy_lab.common.log.BasicLogManager;
import org.sosy_lab.common.log.LogManager;
import org.sosy_lab.java_smt.SolverContextFactory;
import org.sosy_lab.java_smt.SolverContextFactory.Solvers;
import org.sosy_lab.java_smt.api.BooleanFormula;
import org.sosy_lab.java_smt.api.BooleanFormulaManager;
import org.sosy_lab.java_smt.api.IntegerFormulaManager;
import org.sosy_lab.java_smt.api.Model.ValueAssignment;
import org.sosy_lab.java_smt.api.NumeralFormula.IntegerFormula;
import org.sosy_lab.java_smt.api.OptimizationProverEnvironment;
import org.sosy_lab.java_smt.api.OptimizationProverEnvironment.OptStatus;
import org.sosy_lab.java_smt.api.SolverContext;
import org.sosy_lab.java_smt.api.SolverContext.ProverOptions;
import org.sosy_lab.java_smt.api.SolverException;

/** Example for optimizing 'x' with some constraints. */
public class MaximizeCo2 {

  public static void main(String... args)
      throws InvalidConfigurationException, SolverException, InterruptedException {
    Configuration config = Configuration.defaultConfiguration();
    LogManager logger = BasicLogManager.create(config);
    ShutdownNotifier notifier = ShutdownNotifier.createDummy();

    // Z3 supports optimization by default.
    // OptiMathSat can also be used (after being installed separately).
    Solvers solver = Solvers.Z3;

    // create solver context
    try (SolverContext context =
            SolverContextFactory.createSolverContext(config, logger, notifier, solver);
        OptimizationProverEnvironment prover =
            context.newOptimizationProverEnvironment(ProverOptions.GENERATE_MODELS)) {
      BooleanFormulaManager bfmgr = context.getFormulaManager().getBooleanFormulaManager();
      IntegerFormulaManager ifmgr = context.getFormulaManager().getIntegerFormulaManager();

      List<List<ValueAssignment>> models = computeCo2(bfmgr, ifmgr, prover);

      logger.log(Level.INFO, Joiner.on("\n").join(models));
    }
  }

  private static List<List<ValueAssignment>> computeCo2(
      BooleanFormulaManager bfmgr,
      IntegerFormulaManager ifmgr,
      OptimizationProverEnvironment prover)
      throws InterruptedException, SolverException {

    IntegerFormula co2 = ifmgr.makeVariable("co2");
    IntegerFormula thousand = ifmgr.makeNumber(1000);
    BooleanFormula fan = bfmgr.makeVariable("fan");

    // Constraint:  CO2 <= 1000
    prover.addConstraint(ifmgr.lessOrEquals(co2, thousand));

    // Rule:  IF CO2 > 1000 THEN Fan = True
    prover.addConstraint(bfmgr.implication(ifmgr.greaterThan(co2, thousand), fan));

    // Environment: not Fan
    prover.addConstraint(bfmgr.not(fan));

    List<List<ValueAssignment>> models = new ArrayList<>();
    int iterations = 0;

    // produce models where CO2 is maximized.
    // the (unused) handle can be used to get an exact value for the formula
    @SuppressWarnings("unused")
    int handle = prover.maximize(co2);

    while (prover.check() != OptStatus.UNSAT && iterations++ <= 5) {
      final ImmutableList<ValueAssignment> modelAssignments = prover.getModelAssignments();
      models.add(modelAssignments);
      prover.addConstraint(bfmgr.not(bfmgr.and(asFormula(modelAssignments))));
    }

    return models;
  }

  private static List<BooleanFormula> asFormula(
      final ImmutableList<ValueAssignment> modelAssignments) {
    final List<BooleanFormula> modelAssignmentsAsFormulas = new ArrayList<>();
    for (ValueAssignment va : modelAssignments) {
      modelAssignmentsAsFormulas.add(va.getAssignmentAsFormula());
    }
    return modelAssignmentsAsFormulas;
  }
}
