/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.iac.cloudformation.checks;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import javax.annotation.Nullable;
import org.sonar.check.Rule;
import org.sonar.iac.cloudformation.checks.AbstractResourceCheck;
import org.sonar.iac.common.api.checks.CheckContext;
import org.sonar.iac.common.api.tree.HasTextRange;
import org.sonar.iac.common.api.tree.PropertyTree;
import org.sonar.iac.common.api.tree.Tree;
import org.sonar.iac.common.checks.PropertyUtils;
import org.sonar.iac.common.checks.TextUtils;
import org.sonar.iac.common.yaml.tree.MappingTree;
import org.sonar.iac.common.yaml.tree.SequenceTree;
import org.sonar.iac.common.yaml.tree.TupleTree;
import org.sonar.iac.common.yaml.tree.YamlTree;

@Rule(key="S6258")
public class DisabledLoggingCheck
extends AbstractResourceCheck {
    private static final String MESSAGE = "Make sure that disabling logging is safe here.";
    private static final List<String> MSK_LOGGER = Arrays.asList("CloudWatchLogs", "Firehose", "S3");
    private static final String MESSAGE_OMITTING_FORMAT = "Omitting %s makes logs incomplete. Make sure it is safe here.";
    private static final String ENABLED = "Enabled";
    private static final String ENABLE_CLOUDWATCH_LOGS_EXPORTS_KEY = "EnableCloudwatchLogsExports";

    @Override
    protected void checkResource(CheckContext ctx, AbstractResourceCheck.Resource resource) {
        if (DisabledLoggingCheck.isS3Bucket(resource)) {
            DisabledLoggingCheck.checkS3Bucket(ctx, resource);
        } else if (resource.isType("AWS::ApiGateway::Stage")) {
            DisabledLoggingCheck.checkApiGatewayStage(ctx, resource);
        } else if (resource.isType("AWS::ApiGatewayV2::Stage")) {
            DisabledLoggingCheck.checkApiGatewayV2Stage(ctx, resource);
        } else if (resource.isType("AWS::MSK::Cluster")) {
            DisabledLoggingCheck.checkMskCluster(ctx, resource);
        } else if (resource.isType("AWS::Neptune::DBCluster")) {
            DisabledLoggingCheck.checkNeptuneDbCluster(ctx, resource);
        } else if (resource.isType("AWS::DocDB::DBCluster")) {
            DisabledLoggingCheck.checkDocDbCluster(ctx, resource);
        } else if (resource.isType("AWS::AmazonMQ::Broker")) {
            DisabledLoggingCheck.checkAmazonMQBroker(ctx, resource);
        } else if (resource.isType("AWS::Redshift::Cluster")) {
            DisabledLoggingCheck.checkRedshiftCluster(ctx, resource);
        } else if (resource.isType("AWS::Elasticsearch::Domain") || resource.isType("AWS::OpenSearchService::Domain")) {
            DisabledLoggingCheck.checkSearchDomain(ctx, resource);
        } else if (resource.isType("AWS::CloudFront::Distribution")) {
            DisabledLoggingCheck.checkCloudFrontDistribution(ctx, resource);
        } else if (resource.isType("AWS::ElasticLoadBalancing::LoadBalancer")) {
            DisabledLoggingCheck.checkElasticLoadBalancer(ctx, resource);
        } else if (resource.isType("AWS::ElasticLoadBalancingV2::LoadBalancer")) {
            DisabledLoggingCheck.checkElasticLoadBalancerV2(ctx, resource);
        }
    }

    private static void checkS3Bucket(CheckContext ctx, AbstractResourceCheck.Resource resource) {
        YamlTree properties = resource.properties();
        if (PropertyUtils.value((Tree)properties, "LoggingConfiguration").isEmpty() && !DisabledLoggingCheck.isMaybeLoggingBucket(properties)) {
            ctx.reportIssue(resource.type(), DisabledLoggingCheck.omittingMessage("LoggingConfiguration"));
        }
    }

    private static boolean isMaybeLoggingBucket(@Nullable YamlTree properties) {
        Optional<Tree> acl = PropertyUtils.value((Tree)properties, "AccessControl");
        if (acl.isPresent()) {
            Optional<String> scalarValue = TextUtils.getValue(acl.get());
            return scalarValue.map(s -> s.equals("LogDeliveryWrite")).orElse(true);
        }
        return false;
    }

    private static void checkApiGatewayStage(CheckContext ctx, AbstractResourceCheck.Resource resource) {
        YamlTree properties = resource.properties();
        PropertyUtils.value((Tree)properties, "TracingEnabled").ifPresentOrElse(f -> DisabledLoggingCheck.reportOnFalse(ctx, f), () -> DisabledLoggingCheck.reportResource(ctx, resource, DisabledLoggingCheck.omittingMessage("TracingEnabled")));
        DisabledLoggingCheck.reportOnMissingProperty(ctx, properties, "AccessLogSetting", resource.type());
    }

    private static void checkApiGatewayV2Stage(CheckContext ctx, AbstractResourceCheck.Resource resource) {
        DisabledLoggingCheck.reportOnMissingProperty(ctx, resource.properties(), "AccessLogSettings", resource.type());
    }

    private static void checkMskCluster(CheckContext ctx, AbstractResourceCheck.Resource resource) {
        PropertyUtils.get((Tree)resource.properties(), "LoggingInfo").ifPresentOrElse(info -> PropertyUtils.get(info.value(), "BrokerLogs").ifPresentOrElse(logs -> DisabledLoggingCheck.checkMskLogs(ctx, logs), () -> ctx.reportIssue(info.key(), DisabledLoggingCheck.omittingMessage("BrokerLogs"))), () -> DisabledLoggingCheck.reportResource(ctx, resource, DisabledLoggingCheck.omittingMessage("LoggingInfo")));
    }

    private static void checkMskLogs(CheckContext ctx, PropertyTree logs) {
        if (MSK_LOGGER.stream().noneMatch(name -> PropertyUtils.value(logs.value(), name).filter(DisabledLoggingCheck::isLogEnabled).isPresent())) {
            ctx.reportIssue(logs.key(), MESSAGE);
        }
    }

    private static boolean isLogEnabled(Tree logger) {
        return PropertyUtils.value(logger, ENABLED).filter(TextUtils::isValueFalse).isEmpty();
    }

    private static void checkNeptuneDbCluster(CheckContext ctx, AbstractResourceCheck.Resource resource) {
        PropertyUtils.value((Tree)resource.properties(), ENABLE_CLOUDWATCH_LOGS_EXPORTS_KEY).ifPresentOrElse(exportsValue -> {
            if (exportsValue instanceof SequenceTree && ((SequenceTree)exportsValue).elements().isEmpty()) {
                ctx.reportIssue((HasTextRange)exportsValue, MESSAGE);
            }
        }, () -> DisabledLoggingCheck.reportResource(ctx, resource, DisabledLoggingCheck.omittingMessage(ENABLE_CLOUDWATCH_LOGS_EXPORTS_KEY)));
    }

    private static void checkDocDbCluster(CheckContext ctx, AbstractResourceCheck.Resource resource) {
        PropertyUtils.get((Tree)resource.properties(), ENABLE_CLOUDWATCH_LOGS_EXPORTS_KEY).ifPresentOrElse(exportsProperty -> {
            if (exportsProperty.value() instanceof SequenceTree && DisabledLoggingCheck.containsOnlyStringsWithoutAudit((SequenceTree)exportsProperty.value())) {
                ctx.reportIssue(exportsProperty.key(), MESSAGE);
            }
        }, () -> DisabledLoggingCheck.reportResource(ctx, resource, DisabledLoggingCheck.omittingMessage(ENABLE_CLOUDWATCH_LOGS_EXPORTS_KEY)));
    }

    private static boolean containsOnlyStringsWithoutAudit(SequenceTree exports) {
        return exports.elements().stream().allMatch(export -> export.metadata().tag().endsWith("str") && TextUtils.isValue(export, "audit").isFalse());
    }

    private static void checkAmazonMQBroker(CheckContext ctx, AbstractResourceCheck.Resource resource) {
        PropertyUtils.get((Tree)resource.properties(), "Logs").ifPresentOrElse(logs -> {
            if (logs.value() instanceof MappingTree && DisabledLoggingCheck.containsOnlyFalse((MappingTree)logs.value())) {
                ctx.reportIssue(logs.key(), MESSAGE);
            }
        }, () -> DisabledLoggingCheck.reportResource(ctx, resource, MESSAGE));
    }

    private static boolean containsOnlyFalse(MappingTree logs) {
        return logs.elements().stream().map(TupleTree::value).allMatch(TextUtils::isValueFalse);
    }

    private static void checkRedshiftCluster(CheckContext ctx, AbstractResourceCheck.Resource resource) {
        if (PropertyUtils.isMissing(resource.properties(), "LoggingProperties")) {
            DisabledLoggingCheck.reportResource(ctx, resource, DisabledLoggingCheck.omittingMessage("LoggingProperties"));
        }
    }

    private static void checkSearchDomain(CheckContext ctx, AbstractResourceCheck.Resource resource) {
        PropertyUtils.get((Tree)resource.properties(), "LogPublishingOptions").ifPresentOrElse(logs -> DisabledLoggingCheck.checkEnabledAuditLogAvailability(ctx, logs), () -> DisabledLoggingCheck.reportResource(ctx, resource, DisabledLoggingCheck.omittingMessage("LogPublishingOptions")));
    }

    private static void checkEnabledAuditLogAvailability(CheckContext ctx, PropertyTree logs) {
        PropertyUtils.value(logs.value(), "AUDIT_LOGS").flatMap(v -> PropertyUtils.value(v, ENABLED)).ifPresentOrElse(auditLogsEnable -> DisabledLoggingCheck.reportOnFalse(ctx, auditLogsEnable), () -> ctx.reportIssue(logs.key(), DisabledLoggingCheck.omittingMessage("AUDIT_LOGS")));
    }

    private static void checkCloudFrontDistribution(CheckContext ctx, AbstractResourceCheck.Resource resource) {
        PropertyUtils.get((Tree)resource.properties(), "DistributionConfig").ifPresentOrElse(config -> DisabledLoggingCheck.reportOnMissingProperty(ctx, config.value(), "Logging", config.key()), () -> DisabledLoggingCheck.reportResource(ctx, resource, DisabledLoggingCheck.omittingMessage("DistributionConfig")));
    }

    private static void checkElasticLoadBalancer(CheckContext ctx, AbstractResourceCheck.Resource resource) {
        PropertyUtils.value((Tree)resource.properties(), "AccessLoggingPolicy").ifPresentOrElse(policy -> PropertyUtils.value(policy, ENABLED).ifPresent(e -> DisabledLoggingCheck.reportOnFalse(ctx, e)), () -> DisabledLoggingCheck.reportResource(ctx, resource, DisabledLoggingCheck.omittingMessage("AccessLoggingPolicy")));
    }

    private static void checkElasticLoadBalancerV2(CheckContext ctx, AbstractResourceCheck.Resource resource) {
        PropertyUtils.get((Tree)resource.properties(), "LoadBalancerAttributes").ifPresentOrElse(attributes -> DisabledLoggingCheck.getAccessLogsAttribute(attributes.value()).ifPresentOrElse(value -> DisabledLoggingCheck.reportOnFalse(ctx, value), () -> ctx.reportIssue(attributes.key(), MESSAGE)), () -> DisabledLoggingCheck.reportResource(ctx, resource, DisabledLoggingCheck.omittingMessage("LoadBalancerAttributes")));
    }

    private static Optional<Tree> getAccessLogsAttribute(Tree attributes) {
        if (attributes instanceof SequenceTree) {
            return ((SequenceTree)attributes).elements().stream().filter(attribute -> TextUtils.isValue(PropertyUtils.valueOrNull(attribute, "Key"), "access_logs.s3.enabled").isTrue()).map(attribute -> PropertyUtils.value((Tree)attribute, "Value")).findFirst().orElse(Optional.empty());
        }
        return Optional.empty();
    }

    private static void reportOnMissingProperty(CheckContext ctx, @Nullable Tree properties, String property, Tree raiseOn) {
        if (PropertyUtils.isMissing(properties, property)) {
            ctx.reportIssue(raiseOn, DisabledLoggingCheck.omittingMessage(property));
        }
    }

    private static void reportOnFalse(CheckContext ctx, Tree tree) {
        if (TextUtils.isValueFalse(tree)) {
            ctx.reportIssue(tree, MESSAGE);
        }
    }

    private static String omittingMessage(String missingProperty) {
        return String.format(MESSAGE_OMITTING_FORMAT, missingProperty);
    }
}

