package dotty.tools.scaladoc
package renderers

import util.HTML._
import collection.JavaConverters._
import java.net.URI
import java.net.URL
import dotty.tools.scaladoc.site._
import scala.util.Try
import org.jsoup.Jsoup
import java.nio.file.Paths
import java.nio.file.Path
import java.nio.file.Files
import java.io.File
import dotty.tools.scaladoc.translators.FilterAttributes
import util._
import translators._

enum Resource(val path: String):
  case Text(override val path: String, content: String) extends Resource(path)
  case Classpath(override val path: String, name: String) extends Resource(path)
  case File(override val path: String, file: Path) extends Resource(path)
  case URL(url: String) extends Resource(url)
  case URLToCopy(url: String, dest: String) extends Resource(url)

trait Resources(using ctx: DocContext) extends Locations, Writer:
  private def dynamicJsData =
    val str = jsonObject("filterDefaults" -> jsonObject(
      FilterAttributes.defaultValues.toSeq.map { case  (n, v) => n -> jsonString(v) }:_*
    ))
    Resource.Text("scripts/data.js", s"var scaladocData = $str")

  private def scaladocVersionFile = Resource.Text("scaladoc.version", BuildInfo.version)

  private def projectLogo = ctx.args.projectLogo.toSeq.map { p =>
      val path = Paths.get(p)
      Resource.File(s"project-logo/${path.getFileName()}", path)
  }

  private def dottyRes(path: String) = Resource.Classpath(path, s"dotty_res/$path")

  def linkResources(dri: DRI, resources: Iterable[String], deferJs: Boolean): Iterable[AppliedTag] =
    def fileExtension(url: String): String =
      val param = url.indexOf('?')
      val end = if param < 0 then url.length else param
      val point = url.lastIndexOf('.', end)
      url.substring(point+1, end)
    for res <- resources yield
      fileExtension(res) match
        case "css" => link(rel := "stylesheet", href := resolveLink(dri, res))
        case "js" => script(`type` := "text/javascript", src := resolveLink(dri, res), if (deferJs) Seq(defer := "true") else Nil)
        case _ => raw("")

  val onlyRenderedResources: Seq[Resource] =
    List(
      "scripts/inkuire.js"
    ).map(dottyRes) ++
    List(
      "scripts/inkuire-worker.js",
      "webfonts/fa-brands-400.eot",
      "webfonts/fa-brands-400.svg",
      "webfonts/fa-brands-400.ttf",
      "webfonts/fa-brands-400.woff",
      "webfonts/fa-brands-400.woff2",
      "webfonts/fa-regular-400.eot",
      "webfonts/fa-regular-400.svg",
      "webfonts/fa-regular-400.ttf",
      "webfonts/fa-regular-400.woff",
      "webfonts/fa-regular-400.woff2",
      "webfonts/fa-solid-900.eot",
      "webfonts/fa-solid-900.svg",
      "webfonts/fa-solid-900.ttf",
      "webfonts/fa-solid-900.woff",
      "webfonts/fa-solid-900.woff2"
    ).map(dottyRes)


  val earlyMemberResources: Seq[Resource] =
    List(
      "scripts/theme.js"
    ).map(dottyRes)

  val memberResources: Seq[Resource] =
    val fromResources = List(
      "styles/nord-light.css",
      "styles/scalastyle.css",
      "styles/colors.css",
      "styles/dotty-icons.css",
      "styles/diagram.css",
      "styles/filter-bar.css",
      "styles/code-snippets.css",
      "styles/searchbar.css",
      "styles/social-links.css",
      "styles/ux.css",
      "styles/versions-dropdown.css",
      "styles/fontawesome.css",
      "hljs/highlight.min.js",
      "hljs/LICENSE",
      "scripts/hljs-scala3.js",
      "scripts/ux.js",
      "scripts/common/component.js",
      "scripts/common/utils.js",
      "scripts/components/FilterBar.js",
      "scripts/components/DocumentableList.js",
      "scripts/components/Input.js",
      "scripts/components/FilterGroup.js",
      "scripts/components/Filter.js",
      "scripts/scaladoc-scalajs.js"
    ).map(dottyRes)

    val urls = List(
      "https://code.jquery.com/jquery-3.5.1.min.js",
      "https://d3js.org/d3.v6.min.js",
      "https://cdn.jsdelivr.net/npm/graphlib-dot@0.6.2/dist/graphlib-dot.min.js",
      "https://cdnjs.cloudflare.com/ajax/libs/dagre-d3/0.6.1/dagre-d3.min.js",
      "https://scastie.scala-lang.org/embedded.js"
    ).map(Resource.URL.apply)


    fromResources ++ urls ++ projectLogo ++ Seq(scaladocVersionFile, dynamicJsData)

  val searchDataPath = "scripts/searchData.js"
  val scastieConfigurationPath = "scripts/scastieConfiguration.js"
  val memberResourcesPaths = Seq(searchDataPath) ++ Seq(scastieConfigurationPath) ++ memberResources.map(_.path)
  val earlyMemberResourcePaths = earlyMemberResources.map(_.path)

  def searchData(pages: Seq[Page]) =
    def flattenToText(signature: Signature): String =
      signature.map {
        case Type(name, dri) => name
        case Plain(s) => s
        case Keyword(s) => s
      }.mkString

    def mkEntry(dri: DRI, name: String, text: String, extensionTarget: String, descr: String, kind: String) = jsonObject(
        "l" -> jsonString(relativeInternalOrAbsoluteExternalPath(dri)),
        "e" -> (if dri.externalLink.isDefined then rawJSON("true") else rawJSON("false")),
        "i" -> jsonString(extensionTarget),
        "n" -> jsonString(name),
        "t" -> jsonString(text),
        "d" -> jsonString(descr),
        "k" -> jsonString(kind)
      )

    def extensionTarget(member: Member): String =
      member.kind match
        case Kind.Extension(on, _) => flattenToText(on.signature)
        case _ => ""

    def processPage(page: Page, pageFQName: List[String]): Seq[(JSON, Seq[String])] =
      val (res, pageName) =  page.content match
        case m: Member if m.kind != Kind.RootPackage =>
          def processMember(member: Member, fqName: List[String]): Seq[(JSON, Seq[String])] =
            val signatureBuilder = ScalaSignatureProvider.rawSignature(member, InlineSignatureBuilder())().asInstanceOf[InlineSignatureBuilder]
            val sig = Signature(Plain(member.name)) ++ signatureBuilder.names.reverse
            val descr = fqName.mkString(".")
            val entry = mkEntry(member.dri, member.name, flattenToText(sig), extensionTarget(member), descr, member.kind.name)
            val children = member
                .membersBy(m => m.kind != Kind.Package && !m.kind.isInstanceOf[Classlike])
                .filter(m => m.origin == Origin.RegularlyDefined && m.inheritedFrom.fold(true)(_.isSourceSuperclassHidden))
            val updatedFqName = fqName :+ member.name
            Seq((entry, updatedFqName)) ++ children.flatMap(processMember(_, updatedFqName))

          (processMember(m, pageFQName), m.name)
        case _ =>
          (Seq((mkEntry(page.link.dri, page.link.name, page.link.name, "", "", "static"), pageFQName)), "")

      val updatedFqName = if !pageName.isEmpty then pageFQName :+ pageName else pageFQName
      res ++ page.children.flatMap(processPage(_, updatedFqName))

    val entries = pages.flatMap(processPage(_, Nil))
    Resource.Text(searchDataPath, s"pages = ${jsonList(entries.map(_._1))};")

  def scastieConfiguration() =
    Resource.Text(scastieConfigurationPath, s"""scastieConfiguration = "${
      ctx.args.scastieConfiguration.replace('"'.toString, """\"""")
    }"""")


  def allResources(pages: Seq[Page]): Seq[Resource] = earlyMemberResources ++ memberResources ++ Seq(
    dottyRes("favicon.ico"),
    dottyRes("fonts/dotty-icons.woff"),
    dottyRes("fonts/dotty-icons.ttf"),
    dottyRes("images/scaladoc_logo.svg"),
    dottyRes("images/scaladoc_logo_dark.svg"),
    dottyRes("images/class.svg"),
    dottyRes("images/class_comp.svg"),
    dottyRes("images/object.svg"),
    dottyRes("images/object_comp.svg"),
    dottyRes("images/trait.svg"),
    dottyRes("images/trait_comp.svg"),
    dottyRes("images/enum.svg"),
    dottyRes("images/enum_comp.svg"),
    dottyRes("images/given.svg"),
    dottyRes("images/method.svg"),
    dottyRes("images/type.svg"),
    dottyRes("images/val.svg"),
    dottyRes("images/package.svg"),
    dottyRes("images/static.svg"),
    dottyRes("images/inkuire.svg"),
    dottyRes("images/github-icon-black.png"),
    dottyRes("images/github-icon-white.png"),
    dottyRes("images/discord-icon-black.png"),
    dottyRes("images/discord-icon-white.png"),
    dottyRes("images/twitter-icon-black.png"),
    dottyRes("images/twitter-icon-white.png"),
    dottyRes("images/gitter-icon-black.png"),
    dottyRes("images/gitter-icon-white.png"),
    searchData(pages),
    scastieConfiguration(),
  )

  def renderResource(resource: Resource): Seq[String] =
    val normalizedPath = resource.path.replace('\\', '/')
    if normalizedPath.endsWith(".html") && apiPaths.contains(normalizedPath) then
      report.error(s"Conflict between resource and API member for $normalizedPath. $pathsConflictResoultionMsg")
      Nil
    else
      resource match
        case Resource.Text(path, content) =>
          Seq(write(path, content))
        case Resource.Classpath(path, name) =>
          getClass.getClassLoader.getResourceAsStream(name) match
            case null =>
              report.error(s"Unable to find $name on classpath")
              Nil
            case is =>
              try Seq(copy(is, path)) finally is.close()
        case Resource.File(path, file) =>
          Seq(copy(file, path))
        case Resource.URL(url) =>
          Nil
        case Resource.URLToCopy(url, dest) =>
          Seq(copy(new URL(url).openStream(), dest))
