/**
 * Copyright (C) 2013-2016 The Rythm Engine project
 * for LICENSE and other details see:
 * https://github.com/rythmengine/rythmengine
 */
package org.rythmengine.extension;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * <code>Transformer</code> could be used to annotate on a public static methods
 * that works on a given object with or without parameters. For example,
 * <p/>
 * <pre><code>@Transformer public static String format(Number number, String formatStr)</code></pre>
 * <p/>
 * <p>When <code>feature.transformer.enabled</code> is set to true (by default
 * it is true), template author can use Transformer to further process an
 * expression value:</p>
 * <p/>
 * <pre><code>@args Date dueDate\n...@dueDate.format("dd/MM/yyyy")</code></pre>
 * <p/>
 * <p>The <code>@dueDate.format("dd/MM/yyyy")</code> in the above example will be
 * transformed into <code>S.format(dueDate, "dd/MM/yyyy")</code> in the generated
 * java source</p>
 * <p/>
 * <p>Note, the above sample code demonstrates a transformer named <code>format</code>, which is
 * built into Rythm engine. However when you want to define your own transformer, you
 * need to use this <code>Transformer</code> annotation to mark on your methods or classes.
 * When the annotation is marked on a class, then all public static methods with return
 * value and at least one parameter will be treated as transformer</p>
 * <p/>
 * <p>You can register them to RythmEngine by {@link org.rythmengine.RythmEngine#registerTransformer(Class[])}
 * method</p>, once you have registered your Java extension methods, the template author can use them
 * in template. Be careful of the name conflict of your Java extension and Rythm's built-in
 * Java extension. You can turn off rythm's built java extension by set "rythm.disableBuiltInTransformer"
 * to <code>true</code></p>
 * <p/>
 * <p>Transformers mechanism is also found in other template engine
 * solutions, but with different names. Freemarker call it
 * <a href="http://freemarker.sourceforge.net/docs/ref_builtins_number.html">built-ins</a>,
 * Velocity called it <a href="http://stackoverflow.com/questions/8820240/how-to-format-numbers-in-velocity-templates">
 * velocity tools</a>. But none of them are as easy to use as Rythm Transformers</p>
 */
@Retention(RetentionPolicy.RUNTIME)
@Target({ElementType.METHOD, ElementType.TYPE})
public @interface Transformer {
    /**
     * The namespace of the transformer. When namespace is presented, the 
     * template author needs to use the namespace to qualify the transformer
     * in the template source. For example, <code>@x.app_myTransformer()</code> 
     * 
     * <p>Default value: "app"</p>
     * 
     * @return the namespace
     */
    String value() default "app";

    /**
     * Once specified the pattern will be used to match the token to see if transformer extension
     * should be waived or not. For example, <tt>@x.escape()</tt> should be treated as <tt>escape</tt> 
     * transformer while <tt>@s().escape(x)</tt> shouldn't because <tt>s\\(\\)</tt> is a waive pattern
     * 
     * @return the waive pattern 
     */
    String waivePattern() default "";

    /**
     * Require passing {@link org.rythmengine.template.ITemplate template} instance as
     * implicit argument (the first parameter)
     * @return true if require passing template instance as the implicit first parameter 
     */
    boolean requireTemplate() default false;

    /**
     * Indicate whether the expression been transformed should be passed to the transformer function
     * as first or last param
     * @return true if the expression been transformed should be passed in as the last param.
     */
    boolean lastParam() default false;
}
