/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.integration.motion_speech;

import java.util.Map;
import java.util.Properties;
import org.robokind.api.common.lifecycle.AbstractLifecycleProvider;
import org.robokind.api.common.lifecycle.utils.DescriptorListBuilder;
import org.robokind.api.motion.Robot;
import org.robokind.api.motion.blending.FrameSource;
import org.robokind.api.speech.SpeechService;
import org.robokind.api.speech.viseme.VisemeBindingManager;
import org.robokind.api.speech.viseme.VisemeEventNotifier;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class VisemeFrameSourceLifecycle extends 
        AbstractLifecycleProvider<FrameSource, VisemeFrameSource> {
    private final static String theVisemeNotifier = "speechService";
    private final static String theVisemeManager = "visemeManager";
    private Robot.Id myRobotId;
    private VisemeEventNotifier myNotifier;
    
    public VisemeFrameSourceLifecycle(
            Robot.Id robotId, String speechServiceId){
        super(new DescriptorListBuilder()
                .dependency(theVisemeNotifier, VisemeEventNotifier.class)
                    .with(SpeechService.PROP_ID, speechServiceId)
                .dependency(theVisemeManager, VisemeBindingManager.class)
                    .with(Robot.PROP_ID, robotId.getRobtIdString())
                .getDescriptors());
        if(robotId == null || speechServiceId == null){
            throw new NullPointerException();
        }
        myRobotId = robotId;
        myRegistrationProperties = new Properties();
        myRegistrationProperties.put(
                Robot.PROP_ID, myRobotId);
    }

    @Override
    protected VisemeFrameSource create(Map<String, Object> services) {
        myNotifier = (VisemeEventNotifier)services.get(theVisemeNotifier);
        VisemeBindingManager viseme = 
                (VisemeBindingManager)services.get(theVisemeManager);
        VisemeFrameSource vfs = 
                new VisemeFrameSource(myRobotId, viseme);
        myNotifier.addListener(vfs.getVisemeListener());
        return vfs;
    }

    @Override
    protected void handleChange(
            String serviceId, Object service, Map<String,Object> dependencies) {
        if(myNotifier != null && myService != null){
            myNotifier.removeListener(myService.getVisemeListener());
        }
        if(theVisemeManager.equals(serviceId)){
            if(service == null){
                if(myService != null){
                    myService = null;
                }
                return;
            }
            myService = new VisemeFrameSource(
                    myRobotId, (VisemeBindingManager)service);
        }else if(theVisemeNotifier.equals(serviceId)){
            myNotifier = (VisemeEventNotifier)service;
        }
        if(myNotifier != null && myService != null){
            myNotifier.addListener(myService.getVisemeListener());
        }
    }

    @Override
    public Class<FrameSource> getServiceClass() {
        return FrameSource.class;
    }
    
}
