/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.animation.messaging;

import org.apache.avro.util.Utf8;
import org.robokind.api.animation.protocol.PlayRequest;
import org.robokind.api.common.config.VersionProperty;
import org.robokind.api.common.utils.Adapter;
import org.robokind.api.common.utils.TimeUtils;
import org.robokind.avrogen.animation.PlayRequestRecord;

/**
 * PlayRequest which wraps a PlayRequestRecord.
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortablePlayRequest implements PlayRequest{
    private PlayRequestRecord myPlayRquestRecord;

    public PortablePlayRequest(PlayRequestRecord playRec){
        if(playRec == null){
            throw new NullPointerException();
        }
        myPlayRquestRecord = playRec;
    }
    
    public PortablePlayRequest(PlayRequest playRequest){
        PlayRequestRecord rec = new PlayRequestRecord();
        rec.sourceId = new Utf8(playRequest.getSourceId());
        rec.destinationId = new Utf8(playRequest.getDestinationId());
        rec.currentTimeMillisec = playRequest.getTimestampMillisecUTC();
        rec.animationName = new Utf8(playRequest.getAnimationName());
        rec.animationVersionNumber = new Utf8(playRequest.getAnimationVersionNumber());
        myPlayRquestRecord = rec;
    }
    
    public PortablePlayRequest(String sourceId, String destId, 
            long curTimeMillisecUTC, String animName, String animVersionNum){
        PlayRequestRecord rec = new PlayRequestRecord();
        rec.sourceId = new Utf8(sourceId);
        rec.destinationId = new Utf8(destId);
        rec.currentTimeMillisec = curTimeMillisecUTC;
        rec.animationName = new Utf8(animName);
        rec.animationVersionNumber = new Utf8(animVersionNum);
        myPlayRquestRecord = rec;
    }
    
    @Override
    public String getSourceId() {
        return myPlayRquestRecord.sourceId.toString();
    }

    @Override
    public String getDestinationId() {
        return myPlayRquestRecord.destinationId.toString();
    }

    @Override
    public long getTimestampMillisecUTC() {
        return myPlayRquestRecord.currentTimeMillisec;
    }

    @Override
    public String getAnimationName() {
        return myPlayRquestRecord.animationName.toString();
    }

    @Override
    public String getAnimationVersionNumber() {
        if(myPlayRquestRecord.animationVersionNumber == null){
            return null;
        }
        return myPlayRquestRecord.animationVersionNumber.toString();
    }
    
    public PlayRequestRecord getRecord(){
        return myPlayRquestRecord;
    }
    
    public static class Factory implements PlayRequestFactory{
        @Override
        public PlayRequest createPlayRequest(
                String clientId, String hostId, VersionProperty animVersion) {
            if(clientId == null || hostId == null || animVersion == null){
                throw new NullPointerException();
            }
            return new PortablePlayRequest(
                    clientId, hostId, TimeUtils.now(), 
                    animVersion.getName(), animVersion.getNumber());
        }
    }
    
    public static class MessageRecordAdapter 
            implements Adapter<PlayRequest, PlayRequestRecord>{
        @Override
        public PlayRequestRecord adapt(PlayRequest a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortablePlayRequest(a).getRecord();
        }
    }
    
    public static class RecordMessageAdapter implements
            Adapter<PlayRequestRecord, PlayRequest>{
        @Override
        public PlayRequest adapt(PlayRequestRecord a) {
            return new PortablePlayRequest(a);
        }
    }
}
