/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.messaging;

import org.robokind.api.common.utils.Adapter;
import org.robokind.api.common.utils.Listener;

/**
 * High-Level Message sender.
 * Uses an Adapter to convert Messages to Records.
 * Uses a RecordSender to send the Records.
 * Notifies Listeners of Messages sent.
 * 
 * @author Matthew Stevenson <www.robokind.org>
 * @param <Msg> type of message to send
 * @param <Rec> type of Record a message is converted to before being sent
 * @param <L> type of Listener the MessageSender will accept
 */
public interface MessageSender<Msg, Rec, L extends Listener<Msg>> {
    /**
     * Sets the RecordSender responsible for sending Records.
     * @param sender the RecordSender to set
     */
    public void setRecordSender(RecordSender<Rec> sender);
    /**
     * Sets the Adapter used to convert Messages to Records.
     * @param adapter  the Adapter to set
     */
    public void setAdapter(Adapter<Msg,Rec> adapter);
    
    /**
     * Initializes and connects the MessageSender, making it ready to send
     * Messages.
     * @throws Exception if there is an error starting
     */
    public void start() throws Exception;
    /**
     * Stops the MessageSender, making it unable to send Messages.
     */
    public void stop();
    
    /**
     * Sends the given Message.
     * @param message the Message to send
     */
    public void sendMessage(Msg message);
    
    /**
     * Adds a Listener to be notified when a Message has been sent.
     * @param listener the Listener to be notified
     */
    public void addMessageListener(L listener);
    /**
     * Removes a Listener from being notified.
     * @param listener the Listener to remove
     */
    public void removeMessageListener(L listener);
}
