/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.messaging;

import org.robokind.api.common.utils.Adapter;
import org.robokind.api.common.utils.Listener;

/**
 * High-Level Message Receiver.
 * Contains a RecordAsyncReceiver for receiving Records.
 * Contains an Adapter for converting Records back to Messages.
 * Notifies Listeners of Messages Received.
 * 
 * @author Matthew Stevenson <www.robokind.org>
 * @param <Msg> type of Message received
 * @param <Rec> type of Record received
 * @param <L> type of Listener notified of new Messages
 */
public interface MessageAsyncReceiver<Msg, Rec, L extends Listener<Msg>> {
    /**
     * Sets the RecordAsyncReceiver used to receive Records.
     * @param service theRecordReceiver to set
     */
    public void setRecordReceiver(RecordAsyncReceiver<Rec> service);
    /**
     * Sets the Adapter used to convert Records to Messages.
     * @param adapter  the Adapter to set
     */
    public void setAdapter(Adapter<Rec,Msg> adapter);
    
    /**
     * Initializes and connects the MessageAsyncReceiver, and begins receiving
     * Messages.
     * @throws Exception if there is an error starting
     */
    public void start() throws Exception;
    /**
     * Pauses a MessageAsyncReceiver.  
     */
    public void pause();
    /**
     * Resumes a paused MessageAsyncReceiver.
     */
    public void resume();
    /**
     * Stops the MessageAsyncReceiver, making it unable to receive Messages.
     */
    public void stop();
    
    /**
     * Adds a Listener to be notified when a Message has been received.
     * @param listener the Listener to be notified
     */
    public void addMessageListener(L listener);
    /**
     * Removes a Listener from being notified.
     * @param listener the Listener to remove
     */
    public void removeMessageListener(L listener);
}
