/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.messaging;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import org.robokind.api.common.utils.Adapter;
import org.robokind.api.common.utils.Listener;

/**
 * Default MessageSender implementation.
 * 
 * @author Matthew Stevenson <www.robokind.org>
 * @param <Msg> type of Message capable of sending
 * @param <Rec> type of Record which is sent
 * @param <L> type of Listener notified
 */
public class DefaultMessageSender<Msg, Rec, L extends Listener<Msg>> 
        implements MessageSender<Msg,Rec,L>{    
    static final Logger theLogger = Logger.getLogger(DefaultMessageSender.class.getName());
    private List<L> myListeners;
    private Adapter<Msg, Rec> myAdapter;
    /**
     * RecordSender used to send Records.  This is set using the 
     * setRecordSender method.
     */
    protected RecordSender<Rec> myRecordSender;
    /**
     * Creates an empty DefaultMessageSender.
     */
    public DefaultMessageSender(){
        myListeners = new ArrayList<L>();
    }
    
    @Override
    public void setRecordSender(RecordSender<Rec> sender){
        myRecordSender = sender;
    }
    
    @Override
    public void setAdapter(Adapter<Msg, Rec> adapter){
        myAdapter = adapter;
    }

    @Override
    public void start() throws Exception {}

    @Override
    public void stop() {}
    
    @Override
    public void sendMessage(Msg message){
        Rec record = getRecord(message);
        if(record == null){
            theLogger.warning(
                    "Adapter returned null Record, unable to send message.");
            return;
        }
        myRecordSender.sendRecord(record);
    }
    
    protected Rec getRecord(Msg message){
        if(message == null){
            throw new NullPointerException();
        }
        if(myRecordSender == null){
            theLogger.warning("No MessageSender, unable to send message.");
            return null;
        }else if(myAdapter == null){
            theLogger.warning("No Message Adapter, unable to send message.");
            return null;
        }
        return myAdapter.adapt(message);
    }
    
    /**
     * Notifies listeners of a Message being sent.
     * @param message Message being sent
     */
    protected void fireMessageEvent(Msg message){
        for(L listener : myListeners){
            listener.handleEvent(message);
        }
    }
    
    @Override
    public void addMessageListener(L listener) {
        if(listener == null){
            return;
        }
        if(!myListeners.contains(listener)){
            myListeners.add(listener);
        }
    }

    @Override
    public void removeMessageListener(L listener) {
        if(listener == null){
            return;
        }
        myListeners.remove(listener);
    }
    
}
