// TODO: license
package org.reaktivity.nukleus.oauth.internal.types;

import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import org.agrona.BitUtil;
import org.agrona.DirectBuffer;
import org.agrona.MutableDirectBuffer;
import org.agrona.concurrent.UnsafeBuffer;

public final class String32FW extends Flyweight {
  private static final int FIELD_SIZE_LENGTH = BitUtil.SIZE_OF_INT;

  private final ByteOrder byteOrder;

  private final DirectBuffer valueRO = new UnsafeBuffer(0L, 0);

  public String32FW() {
    this.byteOrder = ByteOrder.nativeOrder();
  }

  public String32FW(ByteOrder byteOrder) {
    this.byteOrder = byteOrder;
  }

  public String32FW(String value) {
    this(value, StandardCharsets.UTF_8);
  }

  public String32FW(String value, Charset charset) {
    this.byteOrder = ByteOrder.nativeOrder();
    final byte[] encoded = value.getBytes(charset);
    final MutableDirectBuffer buffer = new UnsafeBuffer(new byte[FIELD_SIZE_LENGTH + encoded.length]);
    buffer.putShort(0, (short)(encoded.length & 0xFFFF));
    buffer.putBytes(FIELD_SIZE_LENGTH, encoded);
    wrap(buffer, 0, buffer.capacity());
  }

  @Override
  public int limit() {
    return offset() + FIELD_SIZE_LENGTH + Math.max(length0(), 0);
  }

  public DirectBuffer value() {
    return length0() == -1 ? null : valueRO;
  }

  public String asString() {
    if (maxLimit() == offset() || length0() == -1) {
      return null;
    }
    return buffer().getStringWithoutLengthUtf8(offset() + FIELD_SIZE_LENGTH, length0());
  }

  @Override
  public String32FW tryWrap(DirectBuffer buffer, int offset, int maxLimit) {
    if (null == super.tryWrap(buffer, offset, maxLimit) || offset + FIELD_SIZE_LENGTH > maxLimit() || limit() > maxLimit) {
      return null;
    }
    int length0 = length0();
    if (length0 != -1) {
      valueRO.wrap(buffer, offset + FIELD_SIZE_LENGTH, length0);
    }
    return this;
  }

  @Override
  public String32FW wrap(DirectBuffer buffer, int offset, int maxLimit) {
    super.wrap(buffer, offset, maxLimit);
    checkLimit(offset + FIELD_SIZE_LENGTH, maxLimit);
    checkLimit(limit(), maxLimit);
    int length0 = length0();
    if (length0 != -1) {
      valueRO.wrap(buffer, offset + FIELD_SIZE_LENGTH, length0);
    }
    return this;
  }

  @Override
  public String toString() {
    return maxLimit() == offset() ? "null" : String.format("\"%s\"", asString());
  }

  private int length0() {
    int length = buffer().getInt(offset(), byteOrder) & 0xFFFFFFFF;
    return length < 0 ? -1 : length;
  }

  public static final class Builder extends Flyweight.Builder<String32FW> {
    private boolean valueSet;

    private final ByteOrder byteOrder;

    public Builder() {
      super(new String32FW());
      this.byteOrder = ByteOrder.nativeOrder();
    }

    public Builder(ByteOrder byteOrder) {
      super(new String32FW(byteOrder));
      this.byteOrder = byteOrder;
    }

    @Override
    public Builder wrap(MutableDirectBuffer buffer, int offset, int maxLimit) {
      checkLimit(offset + FIELD_SIZE_LENGTH, maxLimit);
      super.wrap(buffer, offset, maxLimit);
      this.valueSet = false;
      return this;
    }

    public Builder set(String32FW value) {
      if (value == null) {
        int newLimit = offset() + FIELD_SIZE_LENGTH;
        checkLimit(newLimit, maxLimit());
        buffer().putInt(offset(), -1, byteOrder);
        limit(newLimit);
      } else {
        int newLimit = offset() + value.sizeof();
        checkLimit(newLimit, maxLimit());
        buffer().putInt(offset(), value.length0(), byteOrder);
        buffer().putBytes(offset() + 4, value.buffer(), value.offset() + 4, value.length0());
        limit(newLimit);
      }
      valueSet = true;
      return this;
    }

    public Builder set(DirectBuffer srcBuffer, int srcOffset, int length) {
      checkLength(length);
      int offset = offset();
      int newLimit = offset + length + FIELD_SIZE_LENGTH;
      checkLimit(newLimit, maxLimit());
      buffer().putInt(offset, length, byteOrder);
      buffer().putBytes(offset + 4, srcBuffer, srcOffset, length);
      limit(newLimit);
      valueSet = true;
      return this;
    }

    public Builder set(String value, Charset charset) {
      if (value == null) {
        int newLimit = offset() + FIELD_SIZE_LENGTH;
        checkLimit(newLimit, maxLimit());
        buffer().putInt(offset(), -1, byteOrder);
        limit(newLimit);
      } else {
        byte[] charBytes = value.getBytes(charset);
        checkLength(charBytes.length);
        int newLimit = offset() + FIELD_SIZE_LENGTH + charBytes.length;
        checkLimit(newLimit, maxLimit());
        buffer().putInt(offset(), charBytes.length, byteOrder);
        buffer().putBytes(offset() + 4, charBytes);
        limit(newLimit);
      }
      valueSet = true;
      return this;
    }

    private static void checkLength(int length) {
      final int maxLength = Integer.MAX_VALUE - 1;
      if (length > maxLength) {
        final String msg = String.format("length=%d is beyond maximum length=%d", length, maxLength);
        throw new IllegalArgumentException(msg);
      }
    }

    @Override
    public String32FW build() {
      if (!valueSet) {
        set(null, StandardCharsets.UTF_8);
      }
      return super.build();
    }
  }
}
