/*
 * Copyright 2020 Readium Foundation. All rights reserved.
 * Use of this source code is governed by the BSD-style license
 * available in the top-level LICENSE file of the project.
 */

@file:Suppress("DEPRECATION")

package org.readium.r2.navigator.media

import android.media.session.PlaybackState
import android.os.Bundle
import android.support.v4.media.MediaMetadataCompat
import android.support.v4.media.session.MediaControllerCompat
import android.support.v4.media.session.MediaControllerCompat.TransportControls
import android.support.v4.media.session.PlaybackStateCompat
import kotlin.time.Duration
import kotlin.time.Duration.Companion.milliseconds
import kotlin.time.Duration.Companion.seconds
import kotlin.time.ExperimentalTime
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.*
import org.readium.r2.navigator.MediaNavigator
import org.readium.r2.navigator.extensions.normalizeLocator
import org.readium.r2.navigator.extensions.sum
import org.readium.r2.navigator.media.extensions.elapsedPosition
import org.readium.r2.navigator.media.extensions.id
import org.readium.r2.navigator.media.extensions.isPlaying
import org.readium.r2.navigator.media.extensions.publicationId
import org.readium.r2.navigator.media.extensions.resourceHref
import org.readium.r2.navigator.media.extensions.toPlaybackState
import org.readium.r2.shared.DelicateReadiumApi
import org.readium.r2.shared.InternalReadiumApi
import org.readium.r2.shared.publication.*
import org.readium.r2.shared.util.Url
import org.readium.r2.shared.util.mediatype.MediaType
import timber.log.Timber

/**
 * Rate at which the current locator is broadcasted during playback.
 */
private const val playbackPositionRefreshRate: Double = 2.0 // Hz

private val skipForwardInterval: Duration = 30.seconds
private val skipBackwardInterval: Duration = 30.seconds

/**
 * An implementation of [MediaNavigator] using an Android's MediaSession compatible media player.
 */
@Deprecated(
    "Use the new AudioNavigator from the readium-navigator-media-audio module. This class will be removed in a future 3.x release."
)
@InternalReadiumApi
@OptIn(ExperimentalTime::class)
public class MediaSessionNavigator(
    public val publication: Publication,
    public val publicationId: PublicationId,
    public val controller: MediaControllerCompat,
    public var listener: Listener? = null
) : MediaNavigator, CoroutineScope by MainScope() {

    public interface Listener : MediaNavigator.Listener

    /**
     * Indicates whether the media session is loaded with a resource from this [publication]. This
     * is necessary because a single media session could be used to play multiple publications.
     */
    private val isActive: Boolean get() =
        controller.publicationId == publicationId

    // FIXME: ExoPlayer's media session connector doesn't handle the playback speed yet, so we need the player instance for now
    internal var player: MediaPlayer? = null

    private var playWhenReady: Boolean = false
    private var positionBroadcastJob: Job? = null

    private val needsPlaying: Boolean get() =
        playWhenReady && !controller.playbackState.isPlaying

    /**
     * Duration of each reading order resource.
     */
    private val durations: List<Duration?> =
        publication.readingOrder.map { link ->
            link.duration
                ?.takeIf { it > 0 }
                ?.seconds
        }

    /**
     * Total duration of the publication.
     */
    private val totalDuration: Duration? =
        durations.sum().takeIf { it > 0.seconds }

    private val mediaMetadata = MutableStateFlow<MediaMetadataCompat?>(null)
    private val playbackState = MutableStateFlow<PlaybackStateCompat?>(null)
    private val playbackPosition = MutableStateFlow(0.seconds)

    init {
        controller.registerCallback(MediaControllerCallback())

        launch {
            combine(playbackPosition, mediaMetadata, ::createLocator)
                .filterNotNull()
                .collect { _currentLocator.value = it }
        }
    }

    private val transportControls: TransportControls get() = controller.transportControls

    /**
     * Broadcasts the playback position, as long the media is still playing.
     */
    private fun broadcastPlaybackPosition() {
        positionBroadcastJob?.cancel()
        positionBroadcastJob = launch {
            var state = controller.playbackState
            while (isActive && state.state == PlaybackStateCompat.STATE_PLAYING) {
                val newPosition = state.elapsedPosition.milliseconds
                if (playbackPosition.value != newPosition) {
                    playbackPosition.value = newPosition
                }

                delay((1.0 / playbackPositionRefreshRate).seconds)
                state = controller.playbackState
            }
        }
    }

    // MediaControllerCompat.Callback

    private inner class MediaControllerCallback : MediaControllerCompat.Callback() {

        override fun onMetadataChanged(metadata: MediaMetadataCompat?) {
            if (!isActive || metadata?.id == null) return

            mediaMetadata.value = metadata
        }

        override fun onPlaybackStateChanged(state: PlaybackStateCompat?) {
            if (!isActive) return

            playbackState.value = state
            if (state?.state == PlaybackState.STATE_PLAYING) {
                playWhenReady = false
                broadcastPlaybackPosition()
            }
        }

        override fun onSessionEvent(event: String?, extras: Bundle?) {
            super.onSessionEvent(event, extras)

            if (event == MediaService.EVENT_PUBLICATION_CHANGED && extras?.getString(
                    MediaService.EXTRA_PUBLICATION_ID
                ) == publicationId && playWhenReady && needsPlaying
            ) {
                play()
            }
        }
    }

    // Navigator

    private val _currentLocator = MutableStateFlow(
        Locator(href = Url("#")!!, mediaType = MediaType.BINARY)
    )
    override val currentLocator: StateFlow<Locator> get() = _currentLocator.asStateFlow()

    /**
     * Creates a [Locator] from the given media [metadata] and playback [position].
     */
    @Suppress("RedundantSuspendModifier")
    private suspend fun createLocator(position: Duration?, metadata: MediaMetadataCompat?): Locator? {
        val href = metadata?.resourceHref ?: return null
        val index = publication.readingOrder.indexOfFirstWithHref(href) ?: return null
        var locator = publication.locatorFromLink(publication.readingOrder[index]) ?: return null

        if (position != null) {
            val startPosition = durations.slice(0 until index).sum()
            val duration = durations[index]

            locator = locator.copyWithLocations(
                fragments = listOf("t=${position.inWholeSeconds}"),
                progression = duration?.let { position / duration },
                totalProgression = totalDuration?.let { (startPosition + position) / totalDuration }
            )
        }

        return locator
    }

    @OptIn(DelicateReadiumApi::class)
    override fun go(locator: Locator, animated: Boolean): Boolean {
        if (!isActive) return false

        @Suppress("NAME_SHADOWING")
        val locator = publication.normalizeLocator(locator)

        listener?.onJumpToLocator(locator)

        transportControls.playFromMediaId(
            "$publicationId#${locator.href}",
            Bundle().apply {
                putParcelable("locator", locator)
            }
        )
        return true
    }

    override fun go(link: Link, animated: Boolean): Boolean {
        val locator = publication.locatorFromLink(link) ?: return false
        return go(locator, animated)
    }

    @Suppress("UNUSED_PARAMETER")
    public fun goForward(animated: Boolean = true): Boolean {
        if (!isActive) return false

        seekRelative(skipForwardInterval)
        return true
    }

    @Suppress("UNUSED_PARAMETER")
    public fun goBackward(animated: Boolean = true): Boolean {
        if (!isActive) return false

        seekRelative(-skipBackwardInterval)
        return true
    }

    // MediaNavigator

    override val playback: Flow<MediaPlayback> =
        combine(
            mediaMetadata.filterNotNull(),
            playbackState.filterNotNull(),
            playbackPosition.map { it.inWholeMilliseconds }
        ) { metadata, state, positionMs ->
            // FIXME: Since upgrading to the latest flow version, there's a weird crash when combining a `Flow<Duration>`, like `playbackPosition`. Mapping it seems to do the trick.
            // See https://github.com/Kotlin/kotlinx.coroutines/issues/2353
            val position = positionMs.milliseconds

            val index = metadata.resourceHref?.let {
                publication.readingOrder.indexOfFirstWithHref(
                    it
                )
            }
            if (index == null) {
                Timber.e("Can't find resource index in publication for media ID `${metadata.id}`.")
            }

            val duration = index?.let { durations[index] }

            MediaPlayback(
                state = state.toPlaybackState(),

                // FIXME: ExoPlayer's media session connector doesn't handle the playback speed yet, so I used a custom solution until we create our own connector
//                rate = state?.playbackSpeed?.toDouble() ?: 1.0,
                rate = player?.playbackRate ?: 1.0,

                timeline = MediaPlayback.Timeline(
                    position = position.coerceAtMost(duration ?: position),
                    duration = duration,
                    // Buffering is not yet supported, but will be with media2:
                    // https://developer.android.com/reference/androidx/media2/common/SessionPlayer#getBufferedPosition()
                    buffered = null
                )
            )
        }
            .distinctUntilChanged()
            .conflate()

    override val isPlaying: Boolean
        get() = playbackState.value?.isPlaying == true

    override fun setPlaybackRate(rate: Double) {
        if (!isActive) return
        // FIXME: ExoPlayer's media session connector doesn't handle the playback speed yet, so I used a custom solution until we create our own connector
//        transportControls.setPlaybackSpeed(rate.toFloat())
        player?.playbackRate = rate
    }

    override fun play() {
        if (!isActive) {
            playWhenReady = true
            return
        }
        transportControls.play()
    }

    override fun pause() {
        if (!isActive) return
        transportControls.pause()
    }

    override fun playPause() {
        if (!isActive) return

        if (controller.playbackState.isPlaying) {
            transportControls.pause()
        } else {
            transportControls.play()
        }
    }

    override fun stop() {
        if (!isActive) return
        transportControls.stop()
    }

    override fun seekTo(position: Duration) {
        if (!isActive) return

        @Suppress("NAME_SHADOWING")
        val position = position.coerceAtLeast(0.seconds)

        // We overwrite the current position to allow skipping successively several time without
        // having to wait for the playback position to actually update.
        playbackPosition.value = position

        transportControls.seekTo(position.inWholeMilliseconds)
    }

    override fun seekRelative(offset: Duration) {
        if (!isActive) return

        seekTo(playbackPosition.value + offset)
    }
}
