package org.readium.r2.navigator.input

import android.view.KeyEvent as AndroidKeyEvent

/**
 * Represents a keyboard event emitted by a navigator.
 *
 * @param type Nature of the event.
 * @param key Key the user pressed or released.
 * @param modifiers Additional input modifiers for keyboard shortcuts.
 * @param characters Characters generated by the keypress, if any.
 */
public data class KeyEvent(
    val type: Type,
    val key: Key,
    val modifiers: Set<InputModifier>,
    val characters: String?
) {

    public enum class Type {
        Down, Up
    }

    public companion object {
        public operator fun invoke(type: Type, event: AndroidKeyEvent): KeyEvent? {
            return KeyEvent(
                type = type,
                key = Key(event) ?: return null,
                modifiers = inputModifiers(event),
                characters = event.unicodeChar.toChar().toString()
            )
        }
    }
}

/**
 * Represents a physical key on a keyboard.
 *
 * See https://w3c.github.io/uievents-code/
 */
@JvmInline
public value class Key(public val code: String) {

    public companion object {

        // Alphanumeric section

        public val Backquote: Key = Key("Backquote")
        public val Backslash: Key = Key("Backslash")
        public val BracketLeft: Key = Key("BracketLeft")
        public val BracketRight: Key = Key("BracketRight")
        public val Comma: Key = Key("Comma")
        public val Digit0: Key = Key("Digit0")
        public val Digit1: Key = Key("Digit1")
        public val Digit2: Key = Key("Digit2")
        public val Digit3: Key = Key("Digit3")
        public val Digit4: Key = Key("Digit4")
        public val Digit5: Key = Key("Digit5")
        public val Digit6: Key = Key("Digit6")
        public val Digit7: Key = Key("Digit7")
        public val Digit8: Key = Key("Digit8")
        public val Digit9: Key = Key("Digit9")
        public val Equal: Key = Key("Equal")
        public val IntlBackslash: Key = Key("IntlBackslash")
        public val IntlRo: Key = Key("IntlRo")
        public val IntlYen: Key = Key("IntlYen")
        public val KeyA: Key = Key("KeyA")
        public val KeyB: Key = Key("KeyB")
        public val KeyC: Key = Key("KeyC")
        public val KeyD: Key = Key("KeyD")
        public val KeyE: Key = Key("KeyE")
        public val KeyF: Key = Key("KeyF")
        public val KeyG: Key = Key("KeyG")
        public val KeyH: Key = Key("KeyH")
        public val KeyI: Key = Key("KeyI")
        public val KeyJ: Key = Key("KeyJ")
        public val KeyK: Key = Key("KeyK")
        public val KeyL: Key = Key("KeyL")
        public val KeyM: Key = Key("KeyM")
        public val KeyN: Key = Key("KeyN")
        public val KeyO: Key = Key("KeyO")
        public val KeyP: Key = Key("KeyP")
        public val KeyQ: Key = Key("KeyQ")
        public val KeyR: Key = Key("KeyR")
        public val KeyS: Key = Key("KeyS")
        public val KeyT: Key = Key("KeyT")
        public val KeyU: Key = Key("KeyU")
        public val KeyV: Key = Key("KeyV")
        public val KeyW: Key = Key("KeyW")
        public val KeyX: Key = Key("KeyX")
        public val KeyY: Key = Key("KeyY")
        public val KeyZ: Key = Key("KeyZ")
        public val Minus: Key = Key("Minus")
        public val Period: Key = Key("Period")
        public val Quote: Key = Key("Quote")
        public val Semicolon: Key = Key("Semicolon")
        public val Slash: Key = Key("Slash")

        // Function keys

        public val AltLeft: Key = Key("AltLeft")
        public val AltRight: Key = Key("AltRight")
        public val Backspace: Key = Key("Backspace")
        public val CapsLock: Key = Key("CapsLock")
        public val ContextMenu: Key = Key("ContextMenu")
        public val ControlLeft: Key = Key("ControlLeft")
        public val ControlRight: Key = Key("ControlRight")
        public val Enter: Key = Key("Enter")
        public val MetaLeft: Key = Key("MetaLeft")
        public val MetaRight: Key = Key("MetaRight")
        public val ShiftLeft: Key = Key("ShiftLeft")
        public val ShiftRight: Key = Key("ShiftRight")
        public val Space: Key = Key("Space")
        public val Tab: Key = Key("Tab")

        // Function keys (Japanese and Korean keyboards)

        public val Convert: Key = Key("Convert")
        public val KanaMode: Key = Key("KanaMode")
        public val Lang1: Key = Key("Lang1")
        public val Lang2: Key = Key("Lang2")
        public val Lang3: Key = Key("Lang3")
        public val Lang4: Key = Key("Lang4")
        public val Lang5: Key = Key("Lang5")
        public val NonConvert: Key = Key("NonConvert")

        // Control pad section

        public val Delete: Key = Key("Delete")
        public val End: Key = Key("End")
        public val Help: Key = Key("Help")
        public val Home: Key = Key("Home")
        public val Insert: Key = Key("Insert")
        public val PageDown: Key = Key("PageDown")
        public val PageUp: Key = Key("PageUp")

        // Arrow pad section

        public val ArrowDown: Key = Key("ArrowDown")
        public val ArrowLeft: Key = Key("ArrowLeft")
        public val ArrowRight: Key = Key("ArrowRight")
        public val ArrowUp: Key = Key("ArrowUp")

        // Numpad section

        public val NumLock: Key = Key("NumLock")
        public val Numpad0: Key = Key("Numpad0")
        public val Numpad1: Key = Key("Numpad1")
        public val Numpad2: Key = Key("Numpad2")
        public val Numpad3: Key = Key("Numpad3")
        public val Numpad4: Key = Key("Numpad4")
        public val Numpad5: Key = Key("Numpad5")
        public val Numpad6: Key = Key("Numpad6")
        public val Numpad7: Key = Key("Numpad7")
        public val Numpad8: Key = Key("Numpad8")
        public val Numpad9: Key = Key("Numpad9")
        public val NumpadAdd: Key = Key("NumpadAdd")
        public val NumpadBackspace: Key = Key("NumpadBackspace")
        public val NumpadClear: Key = Key("NumpadClear")
        public val NumpadClearEntry: Key = Key("NumpadClearEntry")
        public val NumpadComma: Key = Key("NumpadComma")
        public val NumpadDecimal: Key = Key("NumpadDecimal")
        public val NumpadDivide: Key = Key("NumpadDivide")
        public val NumpadEnter: Key = Key("NumpadEnter")
        public val NumpadEqual: Key = Key("NumpadEqual")
        public val NumpadHash: Key = Key("NumpadHash")
        public val NumpadMemoryAdd: Key = Key("NumpadMemoryAdd")
        public val NumpadMemoryClear: Key = Key("NumpadMemoryClear")
        public val NumpadMemoryRecall: Key = Key("NumpadMemoryRecall")
        public val NumpadMemoryStore: Key = Key("NumpadMemoryStore")
        public val NumpadMemorySubtract: Key = Key("NumpadMemorySubtract")
        public val NumpadMultiply: Key = Key("NumpadMultiply")
        public val NumpadParenLeft: Key = Key("NumpadParenLeft")
        public val NumpadParenRight: Key = Key("NumpadParenRight")
        public val NumpadStar: Key = Key("NumpadStar")
        public val NumpadSubtract: Key = Key("NumpadSubtract")

        // Function section

        public val Escape: Key = Key("Escape")
        public val F1: Key = Key("F1")
        public val F2: Key = Key("F2")
        public val F3: Key = Key("F3")
        public val F4: Key = Key("F4")
        public val F5: Key = Key("F5")
        public val F6: Key = Key("F6")
        public val F7: Key = Key("F7")
        public val F8: Key = Key("F8")
        public val F9: Key = Key("F9")
        public val F10: Key = Key("F10")
        public val F11: Key = Key("F11")
        public val F12: Key = Key("F12")
        public val Fn: Key = Key("Fn")
        public val FnLock: Key = Key("FnLock")
        public val PrintScreen: Key = Key("PrintScreen")
        public val ScrollLock: Key = Key("ScrollLock")
        public val Pause: Key = Key("Pause")

        // Media keys

        public val BrowserBack: Key = Key("BrowserBack")
        public val BrowserFavorites: Key = Key("BrowserFavorites")
        public val BrowserForward: Key = Key("BrowserForward")
        public val BrowserHome: Key = Key("BrowserHome")
        public val BrowserRefresh: Key = Key("BrowserRefresh")
        public val BrowserSearch: Key = Key("BrowserSearch")
        public val BrowserStop: Key = Key("BrowserStop")
        public val Eject: Key = Key("Eject")
        public val LaunchApp1: Key = Key("LaunchApp1")
        public val LaunchApp2: Key = Key("LaunchApp2")
        public val LaunchMail: Key = Key("LaunchMail")
        public val MediaPlayPause: Key = Key("MediaPlayPause")
        public val MediaSelect: Key = Key("MediaSelect")
        public val MediaStop: Key = Key("MediaStop")
        public val MediaTrackNext: Key = Key("MediaTrackNext")
        public val MediaTrackPrevious: Key = Key("MediaTrackPrevious")
        public val Power: Key = Key("Power")
        public val Sleep: Key = Key("Sleep")
        public val AudioVolumeDown: Key = Key("AudioVolumeDown")
        public val AudioVolumeMute: Key = Key("AudioVolumeMute")
        public val AudioVolumeUp: Key = Key("AudioVolumeUp")
        public val WakeUp: Key = Key("WakeUp")

        public operator fun invoke(event: AndroidKeyEvent): Key? =
            when (event.keyCode) {
                AndroidKeyEvent.KEYCODE_DEL -> Backspace
                AndroidKeyEvent.KEYCODE_ENTER -> Enter
                AndroidKeyEvent.KEYCODE_FORWARD_DEL -> Delete
                AndroidKeyEvent.KEYCODE_SPACE -> Space
                AndroidKeyEvent.KEYCODE_TAB -> Tab

                AndroidKeyEvent.KEYCODE_0 -> Digit0
                AndroidKeyEvent.KEYCODE_1 -> Digit1
                AndroidKeyEvent.KEYCODE_2 -> Digit2
                AndroidKeyEvent.KEYCODE_3 -> Digit3
                AndroidKeyEvent.KEYCODE_4 -> Digit4
                AndroidKeyEvent.KEYCODE_5 -> Digit5
                AndroidKeyEvent.KEYCODE_6 -> Digit6
                AndroidKeyEvent.KEYCODE_7 -> Digit7
                AndroidKeyEvent.KEYCODE_8 -> Digit8
                AndroidKeyEvent.KEYCODE_9 -> Digit9

                AndroidKeyEvent.KEYCODE_A -> KeyA
                AndroidKeyEvent.KEYCODE_B -> KeyB
                AndroidKeyEvent.KEYCODE_C -> KeyC
                AndroidKeyEvent.KEYCODE_D -> KeyD
                AndroidKeyEvent.KEYCODE_E -> KeyE
                AndroidKeyEvent.KEYCODE_F -> KeyF
                AndroidKeyEvent.KEYCODE_G -> KeyG
                AndroidKeyEvent.KEYCODE_H -> KeyH
                AndroidKeyEvent.KEYCODE_I -> KeyI
                AndroidKeyEvent.KEYCODE_J -> KeyJ
                AndroidKeyEvent.KEYCODE_K -> KeyK
                AndroidKeyEvent.KEYCODE_L -> KeyL
                AndroidKeyEvent.KEYCODE_M -> KeyM
                AndroidKeyEvent.KEYCODE_N -> KeyN
                AndroidKeyEvent.KEYCODE_O -> KeyO
                AndroidKeyEvent.KEYCODE_P -> KeyP
                AndroidKeyEvent.KEYCODE_Q -> KeyQ
                AndroidKeyEvent.KEYCODE_R -> KeyR
                AndroidKeyEvent.KEYCODE_S -> KeyS
                AndroidKeyEvent.KEYCODE_T -> KeyT
                AndroidKeyEvent.KEYCODE_U -> KeyU
                AndroidKeyEvent.KEYCODE_V -> KeyV
                AndroidKeyEvent.KEYCODE_W -> KeyW
                AndroidKeyEvent.KEYCODE_X -> KeyX
                AndroidKeyEvent.KEYCODE_Y -> KeyY
                AndroidKeyEvent.KEYCODE_Z -> KeyZ

                AndroidKeyEvent.KEYCODE_APOSTROPHE -> Quote
                AndroidKeyEvent.KEYCODE_BACKSLASH -> Backslash
                AndroidKeyEvent.KEYCODE_GRAVE -> Backquote
                AndroidKeyEvent.KEYCODE_COMMA -> Comma
                AndroidKeyEvent.KEYCODE_EQUALS -> Equal
                AndroidKeyEvent.KEYCODE_LEFT_BRACKET -> BracketLeft
                AndroidKeyEvent.KEYCODE_MINUS -> Minus
                AndroidKeyEvent.KEYCODE_PERIOD -> Period
                AndroidKeyEvent.KEYCODE_RIGHT_BRACKET -> BracketRight
                AndroidKeyEvent.KEYCODE_SEMICOLON -> Semicolon
                AndroidKeyEvent.KEYCODE_SLASH -> Slash

                AndroidKeyEvent.KEYCODE_NUM_LOCK -> NumLock
                AndroidKeyEvent.KEYCODE_NUMPAD_0 -> Numpad0
                AndroidKeyEvent.KEYCODE_NUMPAD_1 -> Numpad1
                AndroidKeyEvent.KEYCODE_NUMPAD_2 -> Numpad2
                AndroidKeyEvent.KEYCODE_NUMPAD_3 -> Numpad3
                AndroidKeyEvent.KEYCODE_NUMPAD_4 -> Numpad4
                AndroidKeyEvent.KEYCODE_NUMPAD_5 -> Numpad5
                AndroidKeyEvent.KEYCODE_NUMPAD_6 -> Numpad6
                AndroidKeyEvent.KEYCODE_NUMPAD_7 -> Numpad7
                AndroidKeyEvent.KEYCODE_NUMPAD_8 -> Numpad8
                AndroidKeyEvent.KEYCODE_NUMPAD_9 -> Numpad9
                AndroidKeyEvent.KEYCODE_NUMPAD_ADD -> NumpadAdd
                AndroidKeyEvent.KEYCODE_NUMPAD_COMMA -> NumpadComma
                AndroidKeyEvent.KEYCODE_NUMPAD_DIVIDE -> NumpadDivide
                AndroidKeyEvent.KEYCODE_NUMPAD_DOT -> NumpadDecimal
                AndroidKeyEvent.KEYCODE_NUMPAD_ENTER -> NumpadEnter
                AndroidKeyEvent.KEYCODE_NUMPAD_EQUALS -> NumpadEqual
                AndroidKeyEvent.KEYCODE_NUMPAD_LEFT_PAREN -> NumpadParenLeft
                AndroidKeyEvent.KEYCODE_NUMPAD_MULTIPLY -> NumpadMultiply
                AndroidKeyEvent.KEYCODE_NUMPAD_RIGHT_PAREN -> NumpadParenRight
                AndroidKeyEvent.KEYCODE_NUMPAD_SUBTRACT -> NumpadSubtract

                AndroidKeyEvent.KEYCODE_CAPS_LOCK -> CapsLock
                AndroidKeyEvent.KEYCODE_ESCAPE -> Escape
                AndroidKeyEvent.KEYCODE_FUNCTION -> Fn
                AndroidKeyEvent.KEYCODE_INSERT -> Insert

                AndroidKeyEvent.KEYCODE_DPAD_DOWN -> ArrowDown
                AndroidKeyEvent.KEYCODE_DPAD_LEFT -> ArrowLeft
                AndroidKeyEvent.KEYCODE_DPAD_RIGHT -> ArrowRight
                AndroidKeyEvent.KEYCODE_DPAD_UP -> ArrowUp
                AndroidKeyEvent.KEYCODE_HOME -> Home
                AndroidKeyEvent.KEYCODE_PAGE_DOWN -> PageDown
                AndroidKeyEvent.KEYCODE_PAGE_UP -> PageUp

                AndroidKeyEvent.KEYCODE_F1 -> F1
                AndroidKeyEvent.KEYCODE_F2 -> F2
                AndroidKeyEvent.KEYCODE_F3 -> F3
                AndroidKeyEvent.KEYCODE_F4 -> F4
                AndroidKeyEvent.KEYCODE_F5 -> F5
                AndroidKeyEvent.KEYCODE_F6 -> F6
                AndroidKeyEvent.KEYCODE_F7 -> F7
                AndroidKeyEvent.KEYCODE_F8 -> F8
                AndroidKeyEvent.KEYCODE_F9 -> F9
                AndroidKeyEvent.KEYCODE_F10 -> F10
                AndroidKeyEvent.KEYCODE_F11 -> F11
                AndroidKeyEvent.KEYCODE_F12 -> F12

                AndroidKeyEvent.KEYCODE_MEDIA_NEXT -> MediaTrackNext
                AndroidKeyEvent.KEYCODE_MEDIA_PLAY_PAUSE -> MediaPlayPause
                AndroidKeyEvent.KEYCODE_MEDIA_PREVIOUS -> MediaTrackPrevious
                AndroidKeyEvent.KEYCODE_MEDIA_STOP -> MediaStop
                AndroidKeyEvent.KEYCODE_VOLUME_DOWN -> AudioVolumeDown
                AndroidKeyEvent.KEYCODE_VOLUME_MUTE -> AudioVolumeMute
                AndroidKeyEvent.KEYCODE_VOLUME_UP -> AudioVolumeUp

                AndroidKeyEvent.KEYCODE_ALT_LEFT -> AltLeft
                AndroidKeyEvent.KEYCODE_ALT_RIGHT -> AltRight
                AndroidKeyEvent.KEYCODE_CTRL_LEFT -> ControlLeft
                AndroidKeyEvent.KEYCODE_CTRL_RIGHT -> ControlRight
                AndroidKeyEvent.KEYCODE_META_LEFT -> MetaLeft
                AndroidKeyEvent.KEYCODE_META_RIGHT -> MetaRight
                AndroidKeyEvent.KEYCODE_SHIFT_LEFT -> ShiftLeft
                AndroidKeyEvent.KEYCODE_SHIFT_RIGHT -> ShiftRight

                else -> null
            }
    }
}

private fun inputModifiers(event: android.view.KeyEvent): Set<InputModifier> =
    buildSet {
        if (event.isAltPressed) {
            add(InputModifier.Alt)
        }
        if (event.isCtrlPressed) {
            add(InputModifier.Control)
        }
        if (event.isMetaPressed) {
            add(InputModifier.Meta)
        }
        if (event.isShiftPressed) {
            add(InputModifier.Shift)
        }
    }
