//===----------------------------------------------------------------------===//
// Copyright © 2024-2025 Apple Inc. and the Pkl project authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//===----------------------------------------------------------------------===//

/// A YAML 1.2 compliant YAML parser.
@ModuleInfo { minPklVersion = "0.27.2" }
module pkl.yaml

/// A YAML parser.
///
/// YAML values are mapped to Pkl values as follows:
///
/// | YAML type   | Pkl type                                                |
/// | ----------- | ------------------------------------------------------- |
/// | !!null      | [Null]                                                  |
/// | !!bool      | [Boolean]                                               |
/// | !!int       | [Int]                                                   |
/// | !!float     | [Float]                                                 |
/// | !!string    | [String]                                                |
/// | !!seq       | [Listing]                                               |
/// | !!set       | [Listing]                                               |
/// | !!map       | [Dynamic] or [Mapping] depending on [Parser.useMapping] |
/// | !!binary    | [String]                                                |
/// | !!timestamp | [String]                                                |
///
/// The element order of YAML sequences, sets, and mappings is maintained.
/// YAML binary values are mapped to their original (base64) string representation.
/// YAML timestamp values are mapped to their original string representation.
///
/// If the parser encounters a YAML type not listed above, it fails with an error.
///
/// Known limitations:
/// - [Merge keys](https://yaml.org/type/merge.html) are not supported.
/// - Comments are not retained.
class Parser {
  /// The YAML version used by producers of the parsed documents.
  ///
  /// - `"compat"` - YAML 1.1 _or_ 1.2 (default)
  /// - `"1.1"` - YAML 1.1
  /// - `"1.2"` - YAML 1.2 (Core schema)
  mode: "compat"|"1.1"|"1.2" = "compat"

  /// Determines what the parser produces when parsing YAML `!!map` types.
  ///
  /// If [true], they turn into a [Mapping].
  /// Otherwise, they turn into a [Dynamic].
  ///
  /// If [useMapping] is [false], map properties named `default` will be shadowed by the
  /// built-in [Dynamic.default] property.
  useMapping: Boolean = false

  /// Value converters to apply to parsed values.
  ///
  /// For further information see [PcfRenderer.converters].
  converters: Mapping<Class|String(!isEmpty), (unknown) -> unknown>

  /// The maximum number of aliases for collection nodes.
  ///
  /// Limiting the number of collection aliases prevents
  /// [billion laughs attacks](https://en.wikipedia.org/wiki/Billion_laughs_attack).
  maxCollectionAliases: Int32(isPositive) = 50

  /// Parses the single YAML document contained in [source].
  ///
  /// Throws if an error occurs during parsing, or if [source] contains multiple YAML documents.
  ///
  /// If [source] is a [Resource], the resource URI is included in parse error messages.
  external function parse(source: Resource|String): Value

  /// Parses all YAML documents contained in [source].
  ///
  /// Throws if an error occurs during parsing.
  ///
  /// If [source] is a [Resource], the resource URI is included in parse error messages.
  external function parseAll(source: Resource|String): List<Value>
}

/// Pkl representation of a YAML value.
typealias Value = Null|Boolean|Number|String|Listing|Dynamic|Mapping
