//===----------------------------------------------------------------------===//
// Copyright © 2024-2025 Apple Inc. and the Pkl project authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//===----------------------------------------------------------------------===//

/// A library for statically analyzing Pkl modules.
///
/// These tools differentiate from [pkl:reflect][reflect] in that they parse Pkl modules, but do not
/// execute any code within these modules.
@Since { version = "0.27.0" }
@ModuleInfo { minPklVersion = "0.27.2" }
module pkl.analyze

// used by doc comments
import "pkl:reflect"

/// Given a set of Pkl module URIs, returns a graph of imports declared by these modules.
///
/// The resulting graph includes transitive imports.
external function importGraph(moduleUris: Set<Uri>): ImportGraph

/// The graph of imports declared (directly and transitively) by the modules passed to
/// [importGraph()].
class ImportGraph {
  /// The imports declared within a Pkl program.
  ///
  /// Each entry maps a module URI to the set of imports declared in that module.
  ///
  /// The set of all modules in the graph can be obtained via its [keys][Map.keys].
  imports: Map<Uri, Set<Import>>

  /// Mappings of modules from their in-language URI, to their resolved URI.
  ///
  /// A module's in-language URI is the form used within Pkl source code.
  /// For example, modulepath-based modules have form `modulepath:/path/to/my/module.pkl`.
  ///
  /// A module's resolved URI is the form used to load the module's contents.
  /// The same modulepath module might have form
  /// `jar:file:///path/to/file.zip!/path/to/my/module.pkl` if Pkl run with
  /// `--module-path /path/to/file.zip`.
  ///
  /// Dependency-notation imports, such as `"@myPackage/myModule.pkl"`, are represented as
  /// in-language URIs with scheme `projectpackage:`.
  /// In the case of local project dependenecies, they will be local URIs resolved from the project
  /// file URI (in normal cases, `file:` URIs).
  resolvedImports: Map<Uri, Uri>(keys == imports.keys)
}

/// An import as declared inside a module.
class Import {
  /// The absolute (in-language) URI of the import.
  ///
  /// Dependency notation URIs (such as `import "@foo/bar"`) are resolved to package URIs with
  /// scheme `projectpackage:`.
  uri: Uri
}
