/*
 * Decompiled with CFR 0.152.
 */
package org.pkl.core.util;

import com.oracle.truffle.api.CompilerDirectives;
import java.math.RoundingMode;
import org.pkl.core.util.Nullable;

public final class MathUtils {
    private static final long FLOOR_SQRT_MAX_LONG = 3037000499L;
    private static final long SIGNIFICAND_MASK = 0xFFFFFFFFFFFFFL;
    private static final long IMPLICIT_BIT = 0x10000000000000L;
    private static final double MIN_LONG_AS_DOUBLE = -9.223372036854776E18;
    private static final double MAX_LONG_AS_DOUBLE_PLUS_ONE = 9.223372036854776E18;
    private static final int SIGNIFICAND_BITS = 52;

    private MathUtils() {
    }

    @CompilerDirectives.TruffleBoundary
    public static long roundToLong(double x, RoundingMode mode) {
        double z = MathUtils.roundIntermediate(x, mode);
        MathUtils.checkInRange(-9.223372036854776E18 - z < 1.0 & z < 9.223372036854776E18);
        return (long)z;
    }

    @CompilerDirectives.TruffleBoundary
    public static boolean isMathematicalInteger(double x) {
        return MathUtils.isFinite(x) && (x == 0.0 || 52 - Long.numberOfTrailingZeros(MathUtils.getSignificand(x)) <= StrictMath.getExponent(x));
    }

    public static boolean isPowerOfTwo(long x) {
        return x > 0L && (x & x - 1L) == 0L;
    }

    @CompilerDirectives.TruffleBoundary
    public static boolean isPowerOfTwo(double x) {
        if (x > 0.0 && MathUtils.isFinite(x)) {
            long significand = MathUtils.getSignificand(x);
            return (significand & significand - 1L) == 0L;
        }
        return false;
    }

    @CompilerDirectives.TruffleBoundary
    public static long checkedPow(long b2, int k) {
        MathUtils.checkNonNegative("exponent", k);
        if (b2 >= -2L & b2 <= 2L) {
            return switch ((int)b2) {
                case 0 -> {
                    if (k == 0) {
                        yield 1L;
                    }
                    yield 0L;
                }
                case 1 -> 1L;
                case -1 -> {
                    if ((k & 1) == 0) {
                        yield 1L;
                    }
                    yield -1L;
                }
                case 2 -> {
                    MathUtils.checkNoOverflow(k < 63);
                    yield 1L << k;
                }
                case -2 -> {
                    MathUtils.checkNoOverflow(k < 64);
                    if ((k & 1) == 0) {
                        yield 1L << k;
                    }
                    yield -1L << k;
                }
                default -> throw new AssertionError();
            };
        }
        long accum = 1L;
        while (true) {
            switch (k) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return MathUtils.checkedMultiply(accum, b2);
                }
            }
            if ((k & 1) != 0) {
                accum = MathUtils.checkedMultiply(accum, b2);
            }
            if ((k >>= 1) <= 0) continue;
            MathUtils.checkNoOverflow(-3037000499L <= b2 && b2 <= 3037000499L);
            b2 *= b2;
        }
    }

    @CompilerDirectives.TruffleBoundary
    public static long gcd(long a, long b2) {
        MathUtils.checkNonNegative("a", a);
        MathUtils.checkNonNegative("b", b2);
        if (a == 0L) {
            return b2;
        }
        if (b2 == 0L) {
            return a;
        }
        int aTwos = Long.numberOfTrailingZeros(a);
        a >>= aTwos;
        int bTwos = Long.numberOfTrailingZeros(b2);
        b2 >>= bTwos;
        while (a != b2) {
            long delta = a - b2;
            long minDeltaOrZero = delta & delta >> 63;
            a = delta - minDeltaOrZero - minDeltaOrZero;
            b2 += minDeltaOrZero;
            a >>= Long.numberOfTrailingZeros(a);
        }
        return a << Math.min(aTwos, bTwos);
    }

    @CompilerDirectives.TruffleBoundary
    public static long checkedMultiply(long a, long b2) {
        int leadingZeros = Long.numberOfLeadingZeros(a) + Long.numberOfLeadingZeros(a ^ 0xFFFFFFFFFFFFFFFFL) + Long.numberOfLeadingZeros(b2) + Long.numberOfLeadingZeros(b2 ^ 0xFFFFFFFFFFFFFFFFL);
        if (leadingZeros > 65) {
            return a * b2;
        }
        MathUtils.checkNoOverflow(leadingZeros >= 64);
        MathUtils.checkNoOverflow(a >= 0L | b2 != Long.MIN_VALUE);
        long result = a * b2;
        MathUtils.checkNoOverflow(a == 0L || result / a == b2);
        return result;
    }

    private static double roundIntermediate(double x, RoundingMode mode) {
        if (!MathUtils.isFinite(x)) {
            throw new ArithmeticException("input is infinite or NaN");
        }
        switch (mode) {
            case UNNECESSARY: {
                MathUtils.checkRoundingUnnecessary(MathUtils.isMathematicalInteger(x));
                return x;
            }
            case FLOOR: {
                if (x >= 0.0 || MathUtils.isMathematicalInteger(x)) {
                    return x;
                }
                return (long)x - 1L;
            }
            case CEILING: {
                if (x <= 0.0 || MathUtils.isMathematicalInteger(x)) {
                    return x;
                }
                return (long)x + 1L;
            }
            case DOWN: {
                return x;
            }
            case UP: {
                if (MathUtils.isMathematicalInteger(x)) {
                    return x;
                }
                return (long)x + (long)(x > 0.0 ? 1 : -1);
            }
            case HALF_EVEN: {
                return StrictMath.rint(x);
            }
            case HALF_UP: {
                double z = StrictMath.rint(x);
                if (StrictMath.abs(x - z) == 0.5) {
                    return x + StrictMath.copySign(0.5, x);
                }
                return z;
            }
            case HALF_DOWN: {
                double z = StrictMath.rint(x);
                if (StrictMath.abs(x - z) == 0.5) {
                    return x;
                }
                return z;
            }
        }
        throw new AssertionError();
    }

    private static long getSignificand(double d2) {
        MathUtils.checkArgument(MathUtils.isFinite(d2), "not a normal value");
        int exponent = StrictMath.getExponent(d2);
        long bits = Double.doubleToRawLongBits(d2);
        return exponent == -1023 ? bits << 1 : (bits &= 0xFFFFFFFFFFFFFL) | 0x10000000000000L;
    }

    private static boolean isFinite(double d2) {
        return Math.getExponent(d2) <= 1023;
    }

    private static void checkInRange(boolean condition) {
        if (!condition) {
            throw new ArithmeticException("not in range");
        }
    }

    private static void checkRoundingUnnecessary(boolean condition) {
        if (!condition) {
            throw new ArithmeticException("mode was UNNECESSARY, but rounding was necessary");
        }
    }

    private static void checkArgument(boolean expression, @Nullable Object errorMessage) {
        if (!expression) {
            throw new IllegalArgumentException(String.valueOf(errorMessage));
        }
    }

    private static void checkNonNegative(String role, int x) {
        if (x < 0) {
            throw new IllegalArgumentException(role + " (" + x + ") must be >= 0");
        }
    }

    private static void checkNonNegative(@Nullable String role, long x) {
        if (x < 0L) {
            throw new IllegalArgumentException(role + " (" + x + ") must be >= 0");
        }
    }

    private static void checkNoOverflow(boolean condition) {
        if (!condition) {
            throw new ArithmeticException("overflow");
        }
    }
}

