#!/usr/bin/env pwsh

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

## This script used for bitbucket monorepos default pipeline to trigger custom component pipeline

# Authorize and get access token
$headers = @{
    "Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes("$env:BITBUCKET_CLIENT_ID`:$env:BITBUCKET_SECRET"));
}
$r = $(Invoke-WebRequest -Method "Post" `
        -Headers $headers `
        -Uri "https://bitbucket.org/site/oauth2/access_token" `
        -Body @{grant_type="client_credentials"}).Content | Out-String | ConvertFrom-Json
$accessToken = $r.access_token

# Get latest pipeline
$headers = @{
    "Authorization" = "Bearer $accessToken";
    "Accept" = "application/json";
}
$r = $(Invoke-WebRequest -Method "Get" `
        -Headers $headers `
        -Uri "https://api.bitbucket.org/2.0/repositories/$env:BITBUCKET_REPO_OWNER/$env:BITBUCKET_REPO_SLUG/pipelines/?sort=-created_on&pagelen=100" `
        ).Content | Out-String | ConvertFrom-Json
$isFirstPipeline = $True
if ($r.size -gt 1) {
    $isFirstPipeline = $False
    $latestPipeline = $r.values | Where-Object {$_.build_number -eq $($env:BITBUCKET_BUILD_NUMBER - 1)}
    Write-Host "Latest pipeline build number - $($latestPipeline.build_number)"
}

# Get changed components
$commitsRange = $env:BITBUCKET_COMMIT
if (-not $isFirstPipeline) {
    if ($env:BITBUCKET_COMMIT -ne $latestPipeline.target.commit.hash) {
        $commitsRange = "$env:BITBUCKET_COMMIT..$($latestPipeline.target.commit.hash)"
    }
}
$url = "https://api.bitbucket.org/2.0/repositories/$env:BITBUCKET_REPO_OWNER/$env:BITBUCKET_REPO_SLUG/diffstat/$commitsRange"
Write-Host "Get changed components by url $url"
$r = $(Invoke-WebRequest -Method "Get" `
        -Headers $headers `
        -Uri $url `
        ).Content | Out-String | ConvertFrom-Json
# For removed files use .values.old.path
$nonUniqueUpdatedPathes = @()
foreach ($diffstatNewItem in $r.values) {
    if ($null -ne $diffstatNewItem.new) {
        $nonUniqueUpdatedPathes += $diffstatNewItem.new.path
    } else {
        $nonUniqueUpdatedPathes += $diffstatNewItem.old.path
    }
}
$updatedComponents = $nonUniqueUpdatedPathes | Where-Object {$_.IndexOf("/") -gt 0} | ForEach-Object {$_.Split("/")[0]} | Get-Unique

# Call custom pipelines
foreach ($component in $updatedComponents) {
    # Skip components which not doesn't have pipelines
    $componentPipelineInRepo = Select-String -Path "$PSScriptRoot/../../../../bitbucket-pipelines.yml" -Pattern $component
    if ($null -eq $componentPipelineInRepo) {
        Write-Host "'$component' pipeline is missing."
        continue
    }
    Write-Host "Invoking '$component' pipeline..."
    $headers = @{
        "Authorization" = "Bearer $accessToken";
        "Accept" = "application/json";
    }
    $body = @{
        "target" = @{
            "commit" = @{
                "hash" = $env:BITBUCKET_COMMIT;
                "type" = "commit"
            };
            "selector" = @{
                "type" = "custom";
                "pattern" = $component
            };
            "type" = "pipeline_commit_target"
        }
    }
    $r = $(Invoke-WebRequest -Method "Post" `
            -Headers $headers `
            -Body $($body | ConvertTo-Json) `
            -ContentType "application/json" `
            -Uri "https://api.bitbucket.org/2.0/repositories/$env:BITBUCKET_REPO_OWNER/$env:BITBUCKET_REPO_SLUG/pipelines/" `
            ).Content | Out-String | ConvertFrom-Json
    Write-Output $r
}
