#!/usr/bin/env pwsh

param
(
    [Parameter(Mandatory = $true, Position = 0)]
    [string] $Org,

    [Parameter(Mandatory = $true, Position = 1)]
    [string] $Name,

    [Parameter(Mandatory = $true, Position = 2)]
    [string] $AWSSecretKeyId,

    [Parameter(Mandatory = $true, Position = 3)]
    [string] $AWSSecretAccessKey,

    [Parameter(Mandatory = $true, Position = 4)]
    [string] $AWSS3Bucket,

    [Parameter(Mandatory = $true, Position = 5)]
    [string] $AWSS3BucketFolder,

    [Parameter(Mandatory = $true, Position = 6)]
    [string] $Token
)

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Set base url for web requests
$baseUrl = "https://api.github.com/repos/$Org/$Name/actions/runs"
$headers = @{
    "Accept" = "application/vnd.github.v3+json"
    "Authorization" = "token $Token"
}

# Get latest pipeline info
$latestPipelineData = Invoke-WebRequest -Method 'Get' -Uri "$baseUrl/$env:GITHUB_RUN_ID" -Headers $headers | ConvertFrom-Json

# Get latest pipeline jobs info
$latestPipelineJobsData = Invoke-WebRequest -Method 'Get' -Uri "$baseUrl/$env:GITHUB_RUN_ID/jobs" -Headers $headers | ConvertFrom-Json

# Initialize resulting object
$triggeredBy = $null
if ($null -ne $latestPipelineData.triggering_actor) {
    $triggeredBy = $latestPipelineData.triggering_actor.login
}
$latestPipelineInfo = @{
    "name" = $name
    "full_name" = $latestPipelineData.name
    "date" = $latestPipelineData.created_at
    "pipeline_url" = $latestPipelineData.html_url
    "triggered_by" = $triggeredBy
    "commit_url" = "https://github.com/$org/$Name/commit/$($latestPipelineData.head_commit.id)}"
}

# Get pipeline status
$latestPipelineInfo.status = $latestPipelineJobsData.jobs[1].conclusion

# Calculate job duration
$startedAt = [datetime]$latestPipelineJobsData.jobs[1].started_at
$finishedAt = [datetime]$latestPipelineJobsData.jobs[1].completed_at
$td = New-Timespan -Start $startedAt -End $finishedAt
$latestPipelineInfo.duration = [Math]::Ceiling($td.TotalSeconds)

# Get status of pipeline jobs
$latestPipelineInfo.builded = $null
$latestPipelineInfo.tested = $null
$latestPipelineInfo.packaged = $null
$latestPipelineInfo.published = $null

$jobsDuration = @()

foreach ($job in $latestPipelineJobsData.jobs[1].steps) {
    if ($job.name -match "build") {
        if ($job.conclusion -eq "success") {
            $latestPipelineInfo.builded = $True
        } else {
            $latestPipelineInfo.builded = $False
        }
    }
    elseif ($job.name -match "test") {
        if ($job.conclusion -eq "success") {
            $latestPipelineInfo.tested = $True
        } else {
            $latestPipelineInfo.tested = $False
        }
    }
    elseif ($job.name -match "package") {
        if ($job.conclusion -eq "success") {
            $latestPipelineInfo.packaged = $True
        } else {
            $latestPipelineInfo.packaged = $False
        }
    }
    elseif ($job.name -match "publish") {
        if ($job.conclusion -eq "success") {
            $latestPipelineInfo.published = $True
        } else {
            $latestPipelineInfo.published = $False
        }
    }

    # Get job duration
    $startedAt = [datetime]$job.started_at
    $finishedAt = [datetime]$job.completed_at
    $td = New-Timespan -Start $startedAt -End $finishedAt
    $duration = [Math]::Ceiling($td.TotalSeconds)

    $jobsDuration += @{
        "name" = $job.name
        "duration" = $duration
    }
}

# Use reusable code to update metrics on s3
Update-S3Metrics
