#!/usr/bin/env pwsh

param
(
    [Parameter(Mandatory = $true, Position = 0)]
    [string] $Org,

    [Parameter(Mandatory = $true, Position = 1)]
    [string] $Name,

    [Parameter(Mandatory = $true, Position = 2)]
    [string] $AWSSecretKeyId,

    [Parameter(Mandatory = $true, Position = 3)]
    [string] $AWSSecretAccessKey,

    [Parameter(Mandatory = $true, Position = 4)]
    [string] $AWSS3Bucket,

    [Parameter(Mandatory = $true, Position = 5)]
    [string] $AWSS3BucketFolder
)

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Authorize and get access token
$headers = @{
    "Authorization" = "Basic "+[System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes("$env:BITBUCKET_CLIENT_ID`:$env:BITBUCKET_SECRET"));
}
$r = $(Invoke-WebRequest -Method "Post" `
        -Headers $headers `
        -Uri "https://bitbucket.org/site/oauth2/access_token" `
        -Body @{grant_type="client_credentials"}).Content | Out-String | ConvertFrom-Json
$Token = $r.access_token

# Set base url for web requests
$baseUrl = "https://api.bitbucket.org/2.0/repositories/$Org/$Name"
$headers = @{
    "Accept" = "application/json"
    "Authorization" = "Bearer $Token"
}

# Get pipeline info
$pipelineData = Invoke-WebRequest -Method 'Get' -Uri "$baseUrl/pipelines/$env:BITBUCKET_PIPELINE_UUID" -Headers $headers | ConvertFrom-Json

# Get pipelines steps
$pipelineSteps = Invoke-WebRequest -Method 'Get' -Uri "$baseUrl/pipelines/$env:BITBUCKET_PIPELINE_UUID/steps/" -Headers $headers | ConvertFrom-Json
$authoringStepData = $pipelineSteps.values | Where-Object {$_.name -eq "Authoring stage"}

# Get commit data
$commitData = Invoke-WebRequest -Method 'Get' -Uri "$baseUrl/commit/$env:BITBUCKET_COMMIT" -Headers $headers | ConvertFrom-Json

# Initialize resulting object
$triggeredBy = $null
$triggeredBy = $commitData.author.raw

# $pipelineName = Split-Path -Path (Get-Location) -Leaf # use component folder name as pipeline name 
$latestPipelineInfo = @{
    "name" = $env:COMPONENT
    "full_name" = $env:COMPONENT
    "date" = $pipelineData.created_on
    "pipeline_url" = "https://bitbucket.org/$Org/$Name/pipelines/results/$env:BITBUCKET_BUILD_NUMBER"
    "triggered_by" = $triggeredBy
    "commit_url" = $pipelineData.target.commit.links.html.href
}

# Read measure json file with script results
$measure = Get-Content -Path "$PSScriptRoot/../../measure.json" | ConvertFrom-Json

# Get pipeline status
$publishStatus = $measure | Where-Object {$_.name -eq "publish"}
if ($null -ne $publishStatus -and $publishStatus.successfull) { #based on publish status
    $status = "success"
} else {
    $status = "failure"
}
$latestPipelineInfo.status = $status

# Calculate stage duration
$startedAt = [datetime]$authoringStepData.started_on
$finishedAt = Get-Date
$td = New-Timespan -Start $startedAt -End $finishedAt
$latestPipelineInfo.duration = [Math]::Ceiling($td.TotalSeconds)

# Get status of pipeline jobs
$latestPipelineInfo.builded = $null
$latestPipelineInfo.tested = $null
$latestPipelineInfo.packaged = $null
$latestPipelineInfo.published = $null

$jobsDuration = @()
foreach ($job in $measure) {
    if ($job.name -match "build") {
        $latestPipelineInfo.builded = $job.successfull
    }
    elseif ($job.name -match "test") {
        $latestPipelineInfo.tested = $job.successfull
    }
    elseif ($job.name -match "package") {
        $latestPipelineInfo.packaged = $job.successfull
    }
    elseif ($job.name -match "publish") {
        $latestPipelineInfo.published = $job.successfull
    }

    $jobsDuration += @{
        "name" = $job.name
        "duration" = $job.duration
    }
}

# Change org & name for monorepo
$Org = $Name
$Name = $env:COMPONENT

# Use reusable code to update metrics on s3
Update-S3Metrics
