#!/usr/bin/env pwsh

param
(
    [Parameter(Mandatory = $true, Position = 0)]
    [string] $Org,

    [Parameter(Mandatory = $true, Position = 1)]
    [string] $Name,

    [Parameter(Mandatory = $true, Position = 2)]
    [string] $AWSSecretKeyId,

    [Parameter(Mandatory = $true, Position = 3)]
    [string] $AWSSecretAccessKey,

    [Parameter(Mandatory = $true, Position = 4)]
    [string] $AWSS3Bucket,

    [Parameter(Mandatory = $true, Position = 5)]
    [string] $AWSS3BucketFolder,

    [Parameter(Mandatory = $true, Position = 6)]
    [string] $Token,

    [Parameter(Mandatory = $true, Position = 7)]
    [string] $JenkinsUrl
)

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Get jenkins host from url
$jenkinsHost = $JenkinsUrl.SubString(0, $JenkinsUrl.IndexOf("/job"))

# Verify jenkins cred variables are set
if ($env:JENKINS_USER -eq $null -or $env:JENKINS_PASSWORD -eq $null) {
    Write-Error "JENKINS_USER or JENKINS_PASSWORD environment variables are not set"
}

# Jenkins auth headers
$pair = "$($env:JENKINS_USER):$($env:JENKINS_PASSWORD)"
$encodedCreds = [System.Convert]::ToBase64String([System.Text.Encoding]::ASCII.GetBytes($pair))
$basicAuthValue = "Basic $encodedCreds"
$Headers = @{
    Authorization = $basicAuthValue
}

# Get latest pipeline run id
$r = Invoke-WebRequest -Method 'Get' -Uri "$JenkinsUrl/wfapi/describe" -Headers $Headers
$pipelineDescription = $r.Content | ConvertFrom-Json
$latestPipelineRun = $pipelineDescription.runCount

# Get latest pipeline info
$r = Invoke-WebRequest -Method 'Get' -Uri "$JenkinsUrl/$latestPipelineRun/wfapi/describe" -Headers $Headers
$latestPipelineData = $r.Content | ConvertFrom-Json

# Get latest pipeline stage info
$authoringJobsUrl = $latestPipelineData.stages | Where-Object {$_.name -eq "Authoring"}
$r = Invoke-WebRequest -Method 'Get' -Uri "$jenkinsHost$($authoringJobsUrl._links.self.href)" -Headers $Headers
$latestPipelineJobsData =  $r.Content | ConvertFrom-Json

# Initialize resulting object
$latestPipelineInfo = @{
    "name" = $name
    "full_name" = $name
    "date" = $([datetimeoffset]::FromUnixTimeMilliseconds($latestPipelineData.startTimeMillis).DateTime | Get-Date -Format u)
    "pipeline_url" = $JenkinsUrl
    "triggered_by" = $env:GIT_AUTHOR_NAME
    "commit_url" = $env:GIT_COMMIT
}

# Get pipeline status
$latestPipelineInfo.status = $latestPipelineJobsData.status.ToLower()

# Calculate job duration
$latestPipelineInfo.duration = $latestPipelineJobsData.durationMillis / 1000

# Get status of pipeline jobs
$latestPipelineInfo.builded = $null
$latestPipelineInfo.tested = $null
$latestPipelineInfo.packaged = $null
$latestPipelineInfo.published = $null
$jobsDuration = @()

# Leave only jobs related to script execution
$filteredJobs = $latestPipelineJobsData.stageFlowNodes | Where-Object {$_.name -eq "Shell Script"}
foreach($job in $filteredJobs) {
    if ($job.parameterDescription -match "build") {
        if ($job.status -eq "SUCCESS") {
            $latestPipelineInfo.builded = $True
        } else {
            $latestPipelineInfo.builded = $False
        }
    }
    elseif ($job.parameterDescription -match "test") {
        if ($job.status -eq "SUCCESS") {
            $latestPipelineInfo.tested = $True
        } else {
            $latestPipelineInfo.tested = $False
        }
    }
    elseif ($job.parameterDescription -match "package") {
        if ($job.status -eq "SUCCESS") {
            $latestPipelineInfo.packaged = $True
        } else {
            $latestPipelineInfo.packaged = $False
        }
    }
    elseif ($job.parameterDescription -match "publish") {
        if ($job.status -eq "SUCCESS") {
            $latestPipelineInfo.published = $True
        } else {
            $latestPipelineInfo.published = $False
        }
    }

    # Get job duration
    $jobsDuration += @{
        "name" = $job.parameterDescription.Split("/")[-1]
        "duration" = $job.durationMillis / 1000
    }
}

# Use reusable code to update metrics on s3
Update-S3Metrics
