#!/usr/bin/env pwsh

param
(
    [Parameter(Mandatory = $true, Position = 0)]
    [string] $OrgUrl,

    [Parameter(Mandatory = $true, Position = 1)]
    [string] $Name,

    [Parameter(Mandatory = $true, Position = 2)]
    [string] $AWSSecretKeyId,

    [Parameter(Mandatory = $true, Position = 3)]
    [string] $AWSSecretAccessKey,

    [Parameter(Mandatory = $true, Position = 4)]
    [string] $AWSS3Bucket,

    [Parameter(Mandatory = $true, Position = 5)]
    [string] $AWSS3BucketFolder,

    [Parameter(Mandatory = $true, Position = 6)]
    [string] $Token
)

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Get organization name from url
$Org = $OrgUrl.split("/")[-2]

# Verify azure cred variables are set
if ($env:AZURE_USER -eq $null -or $env:AZURE_PAT -eq $null) {
    Write-Error "AZURE_USER or AZURE_PAT environment variables are not set"
}

# Azure auth headers
$pair = "$($env:AZURE_USER):$($env:AZURE_PAT)"
$encodedCreds = [System.Convert]::ToBase64String([System.Text.Encoding]::ASCII.GetBytes($pair))
$basicAuthValue = "Basic $encodedCreds"
$Headers = @{
    Authorization = $basicAuthValue
}

# Get latest pipeline build id
$r = Invoke-WebRequest -Method 'Get' -Uri "$OrgUrl$Name/_apis/build/builds" -Headers $Headers
$builds = $($r.Content | ConvertFrom-Json).value
$latestPipelineBuildId = $builds[0].id

# Get latest pipeline info
$r = Invoke-WebRequest -Method 'Get' -Uri "$OrgUrl$Name/_apis/build/builds/$latestPipelineBuildId" -Headers $Headers
$latestPipelineData = $r.Content | ConvertFrom-Json

# Get latest pipeline jobs info
$r = Invoke-WebRequest -Method 'Get' -Uri "$OrgUrl$Name/_apis/build/builds/$latestPipelineBuildId/timeline" -Headers $Headers
$latestPipelineJobsData = $($r.Content | ConvertFrom-Json).records

# Get repository id
$r = Invoke-WebRequest -Method 'Get' -Uri "$OrgUrl$Name/_apis/git/repositories" -Headers $Headers
$repos = $($r.Content | ConvertFrom-Json).value
$repoId = $($repos | Where-Object {$_.project.name -eq $name}).id

# Get latest commit id
$r = Invoke-WebRequest -Method 'Get' -Uri "$OrgUrl$Name/_apis/git/repositories/$repoId/commits" -Headers $Headers
$commits = $($r.Content | ConvertFrom-Json).value
$latestCommit = $commits[0]

# Initialize resulting object
$latestPipelineInfo = @{
    "name" = $name
    "full_name" = $name
    "date" = $latestPipelineData.startTime
    "pipeline_url" = $latestPipelineData._links.web.href
    "triggered_by" = $latestCommit.committer.name
    "commit_url" = $latestCommit.remoteUrl
    "status" = "succeeded" # replaced by worst authoring jobs status
}

# Get pipeline status
$latestPipelineInfo.status = $latestPipelineData.status

# Get status of pipeline jobs
$latestPipelineInfo.builded = $null
$latestPipelineInfo.tested = $null
$latestPipelineInfo.packaged = $null
$latestPipelineInfo.published = $null

$jobsDuration = @()

# Filter only autoring jobs by script names
$latestPipelineAuthoringJobsData = @()
$authoringJobs = @("build", "test", "package", "publish", "tag", "clean")
foreach ($authoringJob in $authoringJobs) {
    $latestPipelineAuthoringJobsData += $latestPipelineJobsData | Where-Object {$_.task -ne $null -and $_.name -match " $authoringJob "}
}

foreach($job in $latestPipelineAuthoringJobsData) {
    if ($job.name -match "build") {
        if ($job.result -eq "succeeded") {
            $latestPipelineInfo.builded = $True
        } else {
            $latestPipelineInfo.builded = $False
        }
    }
    elseif ($job.name -match "test") {
        if ($job.result -eq "succeeded") {
            $latestPipelineInfo.tested = $True
        } else {
            $latestPipelineInfo.tested = $False
        }
    }
    elseif ($job.name -match "package") {
        if ($job.result -eq "succeeded") {
            $latestPipelineInfo.packaged = $True
        } else {
            $latestPipelineInfo.packaged = $False
        }
    }
    elseif ($job.name -match "publish") {
        if ($job.result -eq "succeeded") {
            $latestPipelineInfo.published = $True
        } else {
            $latestPipelineInfo.published = $False
        }
    }

    # Get job duration
    $startedAt = [datetime]$job.startTime
    $finishedAt = [datetime]$job.finishTime
    $td = New-Timespan -Start $startedAt -End $finishedAt
    $duration = [Math]::Ceiling($td.TotalSeconds)

    $jobsDuration += @{
        "name" = $job.name
        "duration" = $duration
    }

    if ($latestPipelineInfo.status -ne "failed") {
        $latestPipelineInfo.status = $job.result
    }
}

# Get pipeline duration
$duration = 0
foreach($job in $jobsDuration) {
    $duration += $job.duration
}
$latestPipelineInfo.duration = $duration

# Use reusable code to update metrics on s3
Update-S3Metrics
