#!/usr/bin/env pwsh

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Load support functions
. "$PSScriptRoot/../../common/include.ps1"

# Get component metadata
$component = Read-Component -Path "$PSScriptRoot/../../../component.json"
$package = Get-Content -Path "$PSScriptRoot/../../../package.json" | ConvertFrom-Json

# Get publish artifact
$nodeArtifact = $component.artifacts | Where-Object {$_.type -eq "npm"}

# Skip publish if artifact not set
if ($nodeArtifact -eq $null) {
    Write-Host "Didn't find any artifacts of the type 'npm' in component.json. Publish to npm skipped."
    exit 0
}

# Verify component and package version
if ($component.type -eq "lib") {
    $packageFilePath = If ([bool]$component.psobject.Properties["product"] -and $component.product -ne "") `
        {"$PSScriptRoot/../../../projects/$($component.product)-$($component.name)/package.json"} Else {"$PSScriptRoot/../../../projects/$($component.name)/package.json"}
    $libPackage = Get-Content -Path $packageFilePath | ConvertFrom-Json
    if ($component.version -ne $package.version -or 
        $component.version -ne $libPackage.version -or
        $package.version -ne $libPackage.version ) {
        Write-Error "Versions in component.json, package.json and projects/$($component.name)/package.json do not match"
    }
} else {
    if ($component.version -ne $package.version) {
        Write-Error "Versions in component.json and package.json do not match"
    }
}

# Automatically login to npm registry by npm-cli-login
if (-not [string]::IsNullOrEmpty($env:NPM_USER) -and`
    -not [string]::IsNullOrEmpty($env:NPM_PASS) -and`
    -not [string]::IsNullOrEmpty($env:NPM_EMAIL)) {
    if (npm whoami -ne $env:NPM_USER) {
        Write-Host "Logging to npmjs registry by npm-cli-login..."
        npm-cli-login
    }
} 

# Automatically login to npm registry by putting token to .npmrc
if (-not [string]::IsNullOrEmpty($env:NPM_TOKEN)) {
    Write-Host "Creating project scope .npmrc"
    Set-Content -Path "$PSScriptRoot/../../../.npmrc" -Value '//registry.npmjs.org/:_authToken=${NPM_TOKEN}'
}

Write-Host "Getting $($package.name) versions from npm registry..."

# Check if version exist on npmjs
$npmjsPackageVersionsRaw = npm view $package.name versions
if ($null -ne $npmjsPackageVersionsRaw -and "$($npmjsPackageVersionsRaw.Where({ $_ -ne " " }))"[0] -ne "[") { # if npm returned str instead of array
    $npmjsPackageVersionsRaw = "[ '$npmjsPackageVersionsRaw' ]"
}
$npmjsPackageVersions = $npmjsPackageVersionsRaw | ConvertFrom-Json
if ($npmjsPackageVersions -contains $package.version) {
    Write-Host "Package already exists on npmjs, publish skipped."
} else {
    if ($env:IS_MONOREPO) {
        Set-Location -Path "$PSScriptRoot/../../../"
    }
    if ($component.type -eq "lib") {
        $buildedPackagePath = If ([bool]$component.psobject.Properties["product"] -and $component.product -ne "") `
            {"$PSScriptRoot/../../../dist/$($component.product)-$($component.name)"} Else {"$PSScriptRoot/../../../dist/$($component.name)"}
        # Move project scope .npmrc to dist folder
        Move-Item -Path "$PSScriptRoot/../../../.npmrc" -Destination $buildedPackagePath
        # Change directory
        Set-Location -Path $buildedPackagePath
    }
    # Publish to npm repository
    Write-Host "Publishing $($package.name) to npm registry..."
    npm publish
}

# Check for successfull publish
if ($LastExitCode -ne 0) {
    Write-Error "Error on npm publish."
}
