#!/usr/bin/env pwsh

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Load support functions
. "$PSScriptRoot/../../common/include.ps1"

# Get component metadata
$component = Read-Component -Path "$PSScriptRoot/../../../component.json"

$dockerArtifacts = $component.artifacts | Where-Object {$_.type -eq "docker"}
if ($dockerArtifacts -eq $null) {
    Write-Host "Didn't find any artifacts of the type 'docker' in component.json. Publish to docker skipped."
    exit 0
}

# Define server name
$pos = $component.registry.IndexOf("/")
$server = ""
if ($pos -gt 0) {
    $server = $component.registry.Substring(0, $pos)
}

# Automatically login to server
if (-not [string]::IsNullOrEmpty($env:DOCKER_USER) -and -not [string]::IsNullOrEmpty($env:DOCKER_PASS)) {
    if ($server -eq "") {
        Write-Host "Logging into Docker Hub"
    } else {
        Write-Host "Logging into $server"
    }
    docker login $server -u $env:DOCKER_USER -p $env:DOCKER_PASS
}

# Publish docker images
foreach($dockerArtifact in $dockerArtifacts) {
    $image = "$($component.registry)/$($dockerArtifact.name):$($dockerArtifact.tag)"
    Write-Host "Pushing $image..."
    docker push $image
    # Check image pushed successfully
    if ($LastExitCode -ne 0) {
        Write-Error "Can't push image '$image' to Docker registry. Make sure you use valid credentials in the DOCKER_USER and DOCKER_PASS environment variables. For more information, check package.ps1 logs."
    }
}
