#!/usr/bin/env pwsh

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Load support functions
. "$PSScriptRoot/../../common/include.ps1"

# Get component metadata
$component = Read-Component -Path "$PSScriptRoot/../../../component.json"
$pubSpecVersion = $(Get-Content -Path "$PSScriptRoot/../../../pubspec.yaml") -match "version:" -replace "version: " -replace "'" -replace "`""

# Get publish artifact
$dartArtifact = $component.artifacts | Where-Object {$_.type -eq "dart"}

# Skip publish if artifact not set
if ($dartArtifact -eq $null) {
    Write-Host "Didn't find any artifacts of the type 'dart' in component.json. Publish to pub.dev skipped."
    exit 0
}

# Verify versions in component.json and pubspec.yaml 
if ($component.version -ne $pubSpecVersion) {
    throw "Versions in component.json and pubspec.yaml do not match"
}

# Login to pub.dev
if (-not [string]::IsNullOrEmpty($env:PUB_DEV_PUBLISH_ACCESS_TOKEN) -and `
    -not [string]::IsNullOrEmpty($env:PUB_DEV_PUBLISH_REFRESH_TOKEN) -and `
    -not [string]::IsNullOrEmpty($env:PUB_DEV_PUBLISH_TOKEN_ENDPOINT) -and `
    -not [string]::IsNullOrEmpty($env:PUB_DEV_PUBLISH_EXPIRATION)) {
    $pubCredentialsPath = "$HOME/.config/dart"
    # TODO: add path for windows
    $pubCredentials = @{
        "accessToken" = $env:PUB_DEV_PUBLISH_ACCESS_TOKEN;
        "refreshToken" = $env:PUB_DEV_PUBLISH_REFRESH_TOKEN;
        "idToken" = $env:PUB_DEV_PUBLISH_ID_TOKEN;
        "tokenEndpoint" = $env:PUB_DEV_PUBLISH_TOKEN_ENDPOINT;
        "scopes" = @("https://www.googleapis.com/auth/userinfo.email","openid");
        "expiration" = [long]$env:PUB_DEV_PUBLISH_EXPIRATION
    }
    # Create credentials.json
    if (-not (Test-Path -Path $pubCredentialsPath)) {
        $null = New-Item -ItemType Directory -Force -Path $pubCredentialsPath
    }
    $pubCredentials | ConvertTo-Json | Set-Content -Path "$pubCredentialsPath/pub-credentials.json"
}

Set-Location -Path "$PSScriptRoot/../../../"
Write-Host "Formatting code before publishing..."
dart format "lib" "test"

# Publish to global repository
dart pub get
Write-Host "`nPushing package to pub.dev registry..."
$publishOutput = dart pub publish -f 2>&1

# Verify execution
if ($LASTEXITCODE -ne 0) {
    $publishError = $publishOutput[-1]
    if ("$publishError".IndexOf("already exists") -gt 0) {
        Write-Host "Package $($component.name):$($component.version) already exists on pub.dev. Publish skipped."
        exit 0
    }
    else {
        Write-Error "Publish failed:`n$publishError"
    }
}
