#!/usr/bin/env pwsh

Set-StrictMode -Version latest
$ErrorActionPreference = "Stop"

# Load support functions
. "$PSScriptRoot/../../common/include.ps1"

# Get component metadata
$component = Read-Component -Path "$PSScriptRoot/../../../component.json"

# Login to docker release registry (in case of using private images during pipeline)
if (($null -ne $env:DOCKER_USER) -and ($null -ne $env:DOCKER_PASS)) {
    # Define registry server name
    $pos = $component.registry.IndexOf("/")
    $server = ""
    if ($pos -gt 0) {
        $server = $component.registry.Substring(0, $pos)
    }
    Write-Host "Logging to $server"
    docker login $server -u $env:DOCKER_USER -p $env:DOCKER_PASS
}


# Verify component script exists
if (Test-Path -Path "$PSScriptRoot/../../../build.ps1") {
    try {
        $startedAt = Get-Date
        # Deligate execution to component script
        . "$PSScriptRoot/../../../build.ps1"
    }
    catch {
        Write-Output $_
        $LASTEXITCODE = 1
    }
    finally {
        # Verify script exit code
        if (Test-Path -Path "variable:LASTEXITCODE") {
            # Save script exit code
            $scriptExitCode = $LASTEXITCODE
        } else {
            # Set exit code if script was empty
            $scriptExitCode = 0
        }
        $completedAt = Get-Date

        # Save script result for bitbucket measure
        if ($null -ne $env:BITBUCKET_BUILD_NUMBER) {
            # Read measure file with pipeline steps results
            $measureFileName = "$PSScriptRoot/../../../../measure.json"
            if (Test-Path -Path $measureFileName) {
                $measure = Get-Content -Path $measureFileName | ConvertFrom-Json
                # Fix single item array issue on ubuntu powershell
                if ($measure.Length -eq 1) {
                    $measure = @($measure)
                }
            } else {
                $measure = @()
            }
            # Define is script finished successfully
            if ($scriptExitCode -eq 0) {
                $successfull = $True
            } else {
                $successfull = $False
            }
            # Create result object
            $result = @{
                "name" = "build";
                "successfull" = $successfull;
                "duration" = [Math]::Ceiling($(New-Timespan -Start $startedAt -End $completedAt).TotalSeconds);
            }
            $measure += $result
            # Save script result to file
            ConvertTo-Json $measure | Set-Content -Path $measureFileName     
        }

        # Retain original exit code
        exit $scriptExitCode
    }
}
