/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: SqlWorkItemStore.java 3118 2006-08-30 14:38:36Z jmettraux $
 */

//
// SqlWorkItemStore.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.worklist.impl.swis;

import javax.security.auth.Subject;

import openwfe.org.MapUtils;
import openwfe.org.Application;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.xml.XmlUtils;
import openwfe.org.sql.SqlUtils;
import openwfe.org.sql.ds.OwfeDataSource;
import openwfe.org.time.Time;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.workitem.CodingException;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.worklist.store.StoreException;
import openwfe.org.worklist.impl.AbstractWorkItemStore;


/**
 * Storing workitems in a RDBMS with vanilla SQL.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: SqlWorkItemStore.java 3118 2006-08-30 14:38:36Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class SqlWorkItemStore

    extends AbstractWorkItemStore

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(SqlWorkItemStore.class.getName());

    //
    // CONSTANTS & co

    /**
     * Use this service param key to set at which rate the 'action' table
     * should be polled
     */
    public final static String P_ACTION_FREQUENCY
        = "actionFrequency";

    private final static String DEFAULT_ACTION_FREQUENCY 
        = "2m";

    /**
     * Use this parameter name to tell the store which ActionDaemon
     * it should instantiate and run, if you don't set it, BasicActionDaemon
     * will be used by default.
     */
    public final static String P_ACTION_DAEMON_CLASSNAME
        = "actionDaemonClass";

    //
    // FIELDS

    /**
     * The daemon that polls the db action table to determine if there
     * is something to do with a workitem (proceed, cancel, ...)
     */
    protected ActionDaemon actionDaemon = null;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        initActionDaemon();
    }

    //
    // METHODS from WorkItemStore

    /**
     * Returns the workitem belonging to a certain workflow instance
     * (overrides method in AbstractWorkItemStore)
     */
    public java.util.List findFlowInstance
        (final Subject s,
         final long workflowInstanceId)
    throws 
        StoreException
    {
        OwfeDataSource ds = null;
        try
        {
            ds = SqlUtils.lookupDataSource
                (getContext(),
                 getParams());

            return SqlWorkItemCoder.findFlowInstance
                (ds, workflowInstanceId);
        }
        catch (Exception e)
        {
            throw new StoreException
                ("Failure", e);
        }
        finally
        {
            ds.releaseConnection();
        }
    }

    //
    // METHODS from AbstractWorkItemStore

    protected void storeWorkItem (final InFlowWorkItem wi)
        throws StoreException
    {
        log.debug("storeWorkItem() "+wi.getLastExpressionId());

        try
        {
            getCoder().encode
                (wi, getContext(), getParams());
        }
        catch (CodingException ce)
        {
            throw new StoreException
                ("Failed to store workitem", ce);
        }
    }

    /**
     * This method is public for the SqlWorkItemStore : so that ActionDaemons
     * implemented by third parties may access it.
     */
    public void removeWorkItem (final FlowExpressionId fei)
        throws StoreException
    {
        try
        {
            SqlWorkItemCoder.removeWorkItem
                (fei, getContext(), getParams());
        }
        catch (Exception e)
        {
            throw new StoreException
                ("Failed to remove workitem", e);
        }
    }

    protected InFlowWorkItem retrieveWorkItem 
        (final Subject s, final FlowExpressionId fei)
    throws 
        StoreException
    {
        try
        {
            return (InFlowWorkItem)getCoder()
                .decode(fei, getContext(), getParams());
        }
        catch (CodingException ce)
        {
            throw new StoreException
                ("Failed to retrieve workitem "+fei, ce);
        }
    }

    protected int doCountWorkItems (final Subject s)
        throws StoreException
    {
        OwfeDataSource ds = null;
        try
        {
            ds = SqlUtils.lookupDataSource
                (getContext(),
                 getParams());

            return SqlWorkItemCoder.countWorkItems(ds);
        }
        catch (Exception e)
        {
            throw new StoreException
                ("Failed to count workitems in store", e);
        }
        finally
        {
            ds.releaseConnection();
        }
    }

    protected java.util.List listWorkItems (final Subject s, int limit)
        throws StoreException
    {
        log.debug("listWorkItems()");

        // 2 step process :
        // 
        // 1) gather flowExpressionIds
        // 2) get workitems
        //
        // but this could really get optimized.
        // (but it's not 'swis' primary concern)
        
        OwfeDataSource ds = null;
        try
        {
            //
            // gather flowExpressionIds

            ds = SqlUtils.lookupDataSource
                (getContext(),
                 getParams());

            log.debug("listWorkItems() got datasource");

            final java.util.List feis = SqlWorkItemCoder
                .determineFlowExpressionIds(ds.getConnection(), -1, -1, limit);

            log.debug("listWorkItems() determined flowExpressionIds");

            //
            // get workitems

            final java.util.List result = new java.util.ArrayList(feis.size());
            final java.util.Iterator it = feis.iterator();
            while (it.hasNext())
            {
                final FlowExpressionId fei = (FlowExpressionId)it.next();

                result.add(getCoder().decode
                    (fei, getContext(), getParams()));
            }

            return result;
        }
        catch (Exception e)
        {
            log.warn
                ("Failed to list workitems in "+getName(), e);

            throw new StoreException
                ("Failed to list workitems in "+getName(), e);
        }
        finally
        {
            ds.releaseConnection();
        }
    }

    //
    // METHODS from Service

    /**
     * Status is outputted as a JDOM element. The status is various
     * information about a service activities and state.
     */
    public org.jdom.Element getStatus ()
    {
        org.jdom.Element result = new org.jdom.Element(getName());

        result.addContent(XmlUtils.getClassElt(this));
        result.addContent(XmlUtils.getRevisionElt("$Id: SqlWorkItemStore.java 3118 2006-08-30 14:38:36Z jmettraux $"));

        return result;
    }

    /**
     * Stops this service and takes care of cancelling the actionDaemon.
     */
    public void stop ()
        throws ServiceException
    {
        this.actionDaemon.cancel();
        log.info("actionDaemon stopped.");

        super.stop();
    }

    //
    // METHODS

    /**
     * The ActionDaemon wakes from time to time to check the action table
     * in the database, if an order is found there for a given workitem (id),
     * that order is executed by the daemon.
     * Of course, you can provide your own ActionDaemon implementation.
     */
    protected void initActionDaemon ()
        throws ServiceException
    {
        final String sActionFreq = MapUtils.getAsString
            (getParams(), P_ACTION_FREQUENCY, DEFAULT_ACTION_FREQUENCY);
        final long actionFreq = 
            Time.parseTimeString(sActionFreq);

        final String actionDaemonClassName = MapUtils.getAsString
            (getParams(), 
             P_ACTION_DAEMON_CLASSNAME, 
             BasicActionDaemon.class.getName());

        try
        {
            this.actionDaemon = ActionDaemon.initActionDaemon
                (actionDaemonClassName,
                 getContext(),
                 getParams(),
                 getName());
        }
        catch (Exception e)
        {
            throw new ServiceException
                ("Failed to init and schedule ActionDaemon of class '"+
                 actionDaemonClassName+"'", e);
        }

        Application.getTimer().schedule
            (this.actionDaemon, 05, actionFreq);

        log.info("initActionDaemon() will wake up every "+sActionFreq);
    }

    //
    // STATIC METHODS

}
