/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: SimpleWorkItemStore.java 3118 2006-08-30 14:38:36Z jmettraux $
 */

//
// SimpleWorkItemStore.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.worklist.impl.store;

import javax.security.auth.Subject;

import openwfe.org.MapUtils;
import openwfe.org.Application;
import openwfe.org.AbstractService;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.auth.BasicPrincipal;
import openwfe.org.engine.workitem.CancelItem;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.dispatch.DispatchingException;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.engine.participants.Participant;
import openwfe.org.engine.participants.ParticipantMap;
import openwfe.org.worklist.Header;
import openwfe.org.worklist.HeaderFactory;
import openwfe.org.worklist.WorkListException;
import openwfe.org.worklist.auth.StorePermission;
import openwfe.org.worklist.impl.LastModifiedHeaderComparator;
import openwfe.org.worklist.store.Lock;
import openwfe.org.worklist.store.PutStrategy;
import openwfe.org.worklist.store.GetStrategy;
import openwfe.org.worklist.store.StoreStrategy;
import openwfe.org.worklist.store.WorkItemStore;
import openwfe.org.worklist.store.StoreException;


/**
 * 1.5.0's style store, complete with a put and a get strategy.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: SimpleWorkItemStore.java 3118 2006-08-30 14:38:36Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class SimpleWorkItemStore

    extends AbstractService

    implements WorkItemStore

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(SimpleWorkItemStore.class.getName());

    //
    // CONSTANTS & co

    /**
     * If the parameter 'default' for this store is set to 'true' or 'yes', 
     * the store will be a default store.
     */
    public final static String P_DEFAULT
        = "default";

    /**
     * The 'storage' parameter indicates to this store which
     * storage is should use.
     */
    public final static String P_STORAGE
        = "storage";

    /**
     * If the param 'storage' is not set, the store will look for a
     * storage service simply named 'storage'.
     */
    public final static String DEFAULT_STORAGE
        = "storage";

    /**
     * The param 'putStrategyClass' sets the [put] strategy the store will
     * use when asked to store workitems.
     */
    public final static String P_PUT_STRATEGY_CLASS
        = "putStrategyClass";

    /**
     * The param 'getStrategyClass' sets the [get] strategy the store will
     * use when asked for workitem retrieval.
     */
    public final static String P_GET_STRATEGY_CLASS
        = "getStrategyClass";

    /**
     * Constant for the time parameter named 'lockTimeout' : how many
     * time an idle lock should be maintained on a workitem ?
     */
    public final static String P_LOCK_TIMEOUT
        = "lockTimeout";

    /**
     * Constant for the time parameter named 'unlockFrequency' : how 
     * frequently should the workitem store check for workitem whose
     * locks did time out ?
     */
    public final static String P_UNLOCK_FREQUENCY
        = "unlockFrequency";

    /** 
     * Set to 15m
     */
    public final static String DEFAULT_LOCK_TIMEOUT
        = "15m";

    /**
     * Set to 7m
     */
    public final static String DEFAULT_UNLOCK_FREQUENCY
        = "7m";

    /**
     * Constant for the parameter named 'headerFactory' : a header is a 
     * summary of a workitem for display in a selection list, a header
     * factory is a piece of code that creates a header for a workitem. Use
     * this parameter to indicate the workitem store which header factory
     * (an OpenWFE like an other) it should use.
     */
    public final static String P_HEADER_FACTORY
        = "headerFactory";

    //
    // FIELDS

    private String storageName = null;
    private PutStrategy putStrategy = null;
    private GetStrategy getStrategy = null;

    // lock related fields
    private java.util.Map lockMap = new java.util.HashMap();
    private long lockTimeout = 0;
    private UnlockDaemon unlockDaemon = null;

    private HeaderFactory headerFactoryService = null;

    private String[] acceptedParticipants = null;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        //
        // set storage name
        
        this.storageName = MapUtils.getAsString
            (serviceParams, P_STORAGE, DEFAULT_STORAGE);

        if (log.isDebugEnabled())
        {
            log.debug
                ("init() store '"+getName()+
                 "' is using storage '"+this.storageName+"'");
        }

        //
        // set putStrategy
        
        String classname = MapUtils.getAsString
            (serviceParams, 
             P_PUT_STRATEGY_CLASS, 
             DefaultPutStrategy.class.getName());

        this.putStrategy = (PutStrategy)buildStrategy(classname);

        //
        // set getStrategy

        classname = MapUtils.getAsString
            (serviceParams, 
             P_GET_STRATEGY_CLASS, 
             DefaultGetStrategy.class.getName());

        this.getStrategy = (GetStrategy)buildStrategy(classname);

        //
        // init accepted participants
        
        initAcceptedParticipants();

        //
        // init UnlockDaemon
        
        initUnlockDaemon();

        //
        // init done.

        log.info("init() store '"+getName()+"' ready.");
    }

    private StoreStrategy buildStrategy (final String strategyClassName)
        throws ServiceException
    {
        try
        {
            final StoreStrategy ss = 
                (StoreStrategy)Class.forName(strategyClassName).newInstance();

            ss.init(getContext(), this, getParams(), getStorageName());

            log.info("buildStrategy() of class "+strategyClassName);

            return ss;
        }
        catch (final Exception e)
        {
            throw new ServiceException
                ("Failed to build strategy of class "+strategyClassName, e);
        }
    }

    /**
     * This method is called by the init() method, to determine how the
     * unlock daemon should behave and to start it.
     */
    protected void initUnlockDaemon ()
    {
        this.lockTimeout = MapUtils.getAsTime
            (getParams(), P_LOCK_TIMEOUT, DEFAULT_LOCK_TIMEOUT);

        log.info("lockTimeout set to "+this.lockTimeout+" ms");

        final long unlockFreq = MapUtils.getAsTime
            (getParams(), P_UNLOCK_FREQUENCY, DEFAULT_UNLOCK_FREQUENCY);

        this.unlockDaemon = new UnlockDaemon();

        Application.getTimer().schedule
            (this.unlockDaemon, 60, unlockFreq);

        log.info("UnlockDaemon invoked. Will wake up every "+unlockFreq+" ms");
    }

    /**
     * Establishes a list of participant names that this workitem
     * stores is going to accept when the worklist will iterator
     * among its stores.
     */
    protected void initAcceptedParticipants ()
    {
        final String rawString = (String)getParams().get(P_PARTICIPANTS);
        
        if (rawString == null) 
            this.acceptedParticipants = new String[] {};
        else
            this.acceptedParticipants = rawString.split(", *");
    }

    //
    // GETTERS and SETTERS

    /**
     * Returns the name of the storage service this store works with.
     */
    public String getStorageName ()
    {
        return this.storageName;
    }

    /**
     * Returns the 'input' strategy this store uses.
     */
    public PutStrategy getPutStrategy ()
    {
        return this.putStrategy;
    }

    /**
     * Returns the 'retrieval' strategy this store uses.
     */
    public GetStrategy getGetStrategy ()
    {
        return this.getStrategy;
    }

    //
    // METHODS

    /**
     * Unlocks the given workitem id.
     */
    protected void unlock (final FlowExpressionId workitemId)
        throws StoreException
    {
        if (log.isDebugEnabled())
            log.debug("unlock() for "+workitemId);

        this.lockMap.remove(workitemId);
    }

    /**
     * Returns the header factory this store works with.
     */
    protected HeaderFactory getHeaderFactory ()
    {
        if (this.headerFactoryService == null)
        {
            final String serviceName = MapUtils.getAsString
                (getParams(), 
                 P_HEADER_FACTORY, 
                 openwfe.org.worklist.Definitions.S_HEADER_FACTORY);

            this.headerFactoryService = (HeaderFactory)getContext()
                .get(serviceName);
        }

        return this.headerFactoryService;
    }

    /**
     * Locks a workitem. Internal usage.
     */
    protected void lockWorkItem 
        (final Subject s, final FlowExpressionId workitemId)
    {
        this.lockMap.put(workitemId, new Lock(s, this.lockTimeout));

        if (log.isDebugEnabled())
            log.debug("lockWorkItem() locked "+workitemId);
    }

    //
    // METHODS from WorkItemStore

    /**
     * Returns true if the given workitem is locked.
     */
    public boolean isLocked (final FlowExpressionId workitemId)
    {
        return this.lockMap.keySet().contains(workitemId);
    }

    /**
     * Returns true if the given workitem (id) has been locked by another
     * subject (principal).
     */
    public boolean isLockedBySomeoneElse 
        (final Subject s, final FlowExpressionId workitemId)
    {
        final Lock l = getLock(workitemId);

        if (l == null) return false;

        return ! l.getLocker().equals(s);
    }

    /**
     * Returns the lock set on the workitem, will return null if no lock is set
     * on the workitem.
     */
    public Lock getLock (final FlowExpressionId workitemId)
    {
        return (Lock)this.lockMap.get(workitemId);
    }

    /**
     * Returns the name of the locker of this workitem, or null if the
     * workitem is not locked.
     */
    public String getLockerName (final FlowExpressionId workitemId)
    {
        final Lock l = getLock(workitemId);

        if (l == null) return null;

        return
            BasicPrincipal.getBasicPrincipal(l.getLocker()).getName();
    }

    /**
     * Inserts a workitem in the store.
     * This method takes no Subject as parameter because it is used
     * by listeners, not clients.
     */
    public void store (final InFlowWorkItem wi)
        throws StoreException
    {
        this.putStrategy.put(wi);
    }

    /**
     * 'Cancels' a workitem, either remove it, either flags it as cancelled
     */
    public void cancel (final CancelItem ci)
        throws StoreException
    {
        this.putStrategy.remove(ci.getId());
    }

    /**
     * Delegate works to this workitem store
     */
    public void delegate (final InFlowWorkItem wi)
        throws StoreException
    {
        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "delegate"));

        if (log.isDebugEnabled())
        {
            log.debug
                ("delegate() store '"+getName()+"' received delegated wi "+
                 wi.getLastExpressionId());
        }

        this.putStrategy.put(wi);
    }

    /**
     * Delegate works to this workitem store
     */
    public void delegateToParticipant 
        (final Subject s, final InFlowWorkItem wi, final String participantName)
    throws 
        StoreException
    {
        if (log.isDebugEnabled())
        {
            log.debug
                ("delegateToParticipant() '"+this.getName()+
                 "' to >"+participantName+"<");
        }

        //
        // find participant
        
        final ParticipantMap pMap = openwfe.org.engine.Definitions
            .getParticipantMap(getContext());

        final Participant p = pMap.get(participantName);

        if (p == null)
        {
            throw new StoreException
                ("Cannot delegate to unknown participant '"+
                 participantName+"'");
        }

        //
        // remove from store

        this.remove(s, wi.getLastExpressionId());

        //
        // tag history

        wi.addHistoryItem
            ("subject::"+BasicPrincipal.getBasicPrincipal(s).getName(),
             "Delegated to participant::"+participantName);

        wi.setParticipantName(participantName);
            //
            // else the dispatchment target will not put the wi in
            // the right place.

        //
        // dispatch
        
        try
        {
            p.dispatch(getContext(), wi);
        }
        catch (final Throwable t)
        {
            throw new StoreException
                ("Failed to delegate to participant '"+participantName+"'", t);
        }

        log.debug("delegateToParticipant() dispatching done.");
    }

    /**
     * Updates a workitem in the store
     */
    public void save (final Subject s, final InFlowWorkItem wi)
        throws StoreException
    {
        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "write"));

        synchronized (this) // check impact on perf
        {
            final FlowExpressionId id = wi.getLastExpressionId();

            //
            // is the item locked ?
            
            final Lock lock = (Lock)this.lockMap.get(id);
            if (lock != null)
            {
                Subject locker = lock.getLocker();

                //
                // is the subject the locker ?
                
                if ( ! s.equals(locker))
                {
                    throw new StoreException
                        ("WorkItem '"+id+
                         "' is already locked by someone else.");
                }
            }
            else
            {
                //log.debug("Saving a workitem that was not locked before...");

                throw new StoreException
                    ("Cannot save workitem : you have no lock on it.");
            }

            //
            // proceed

            wi.touch();
                // update wi.lastModified

            this.putStrategy.put(wi);

            lock.touch();
            // leave lock untouched ???
        }
    }

    /**
     * Removes a workitem from the store
     */
    public void remove (final Subject s, final FlowExpressionId expressionId)
        throws StoreException
    {
        //openwfe.org.Utils.debugMap(log, this.lockMap); // DEBUG

        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "write"));

        //
        // is the item locked ?
        
        final Lock lock = (Lock)this.lockMap.get(expressionId);

        if (lock == null)
        {
            throw new StoreException
                ("You cannot remove a workitem when there "+
                 "is no lock on it.");
        }

        final Subject locker = lock.getLocker();

        //
        // is the subject the locker ?
        
        if ( ! s.equals(locker))
        {
            throw new StoreException
                ("WorkItem '"+expressionId+
                 "' is already locked by someone else.");
        }

        //
        // do it

        this.putStrategy.remove(expressionId);

        //
        // remove lock

        unlock(expressionId);

        if (log.isDebugEnabled())
            log.debug(getName()+" remove() removed "+expressionId);
    }

    /**
     * Fetches a workitem from the store.
     */
    public InFlowWorkItem get (final Subject s, final FlowExpressionId fei)
        throws StoreException
    {
        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "read"));

        return this.getStrategy.retrieveWorkItem(s, fei);
    }

    /**
     * Fetches a workitem, the store will tag it as locked.
     */
    public InFlowWorkItem getAndLock 
        (final Subject s, final FlowExpressionId expressionId)
    throws 
        StoreException
    {
        if (log.isDebugEnabled())
            log.debug("getAndLock() from '"+getName()+"' for "+expressionId);

        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "write"));

        //debugLockMap();

        if (isLockedBySomeoneElse(s, expressionId))
        {
            throw new StoreException
                ("WorkItem already locked by someone else");
        }

        synchronized (this)
        {
            //
            // get item
            
            final InFlowWorkItem wi = get(s, expressionId);

            //
            // lock it
            
            lockWorkItem(s, expressionId);

            debugHistory("getAndLock()", wi);

            return wi;
        }
    }

    /*
     * trying to locate Filip's workitem history problem...
     */
    private void debugHistory 
        (final String message, final InFlowWorkItem wi)
    {
        if (wi.getHistory().size() < 1)
        {
            log.debug(message+" [debugHistory] no history...");
            return;
        }

        final StringBuffer sb = new StringBuffer();

        final java.util.Iterator it = wi.getHistory().iterator();

        while (it.hasNext())
        {
            sb.append("   - ").append(it.next().toString());

            if (it.hasNext()) sb.append("\n");
        }

        log.debug(message+" [debugHistory] \n"+sb.toString());
    }

    private void debugLockMap ()
    {
        final StringBuffer sb = new StringBuffer();

        synchronized (this.lockMap)
        {
            final java.util.Iterator it = this.lockMap.keySet().iterator();
            while (it.hasNext())
            {
                final FlowExpressionId id = (FlowExpressionId)it.next();
                final Lock lock = (Lock)this.lockMap.get(id);

                sb.append("  - ");
                sb.append(id);
                sb.append("  <--  ");

                sb.append(openwfe.org.auth.BasicPrincipal
                    .getBasicPrincipal(lock.getLocker()).getName());

                if (it.hasNext()) sb.append("\n");
            }
        }

        log.debug("debugLockMap() :\n"+sb.toString());
    }

    /**
     * Releases the lock on a workitem
     */
    public void release (final Subject s, final FlowExpressionId workItemId)
        throws StoreException
    {
        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "write"));

        synchronized (this)
        {
            //
            // find lock

            final Lock lock = (Lock)this.lockMap.get(workItemId);

            if (lock == null)
            {
                throw new StoreException
                    ("WorkItem is not locked. No need to release it.");
            }

            final Subject locker = lock.getLocker();

            //log.debug("release() locker is "+locker);

            //if (locker == null) return; // success by default

            //if ( ! s.equals(locker))
            if (isNotTheSameLocker(s, locker))
            {
                throw new StoreException
                    ("WorkItem is locked by someone else.");
            }

            //
            // remove lock

            unlock(workItemId);

            if (log.isDebugEnabled())
                log.debug(getName()+" release() released "+workItemId);
        }
    }

    private boolean isNotTheSameLocker 
        (final Subject unlocker, final Subject locker)
    {
        if (locker == null) return false;

        return ! locker.equals(unlocker);
    }

    /**
     * Returns the count of workitems found in this store.
     */
    public int countWorkItems (final Subject s)
        throws StoreException
    {
        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "browse"));

        return this.getStrategy.countWorkItems(s);
    }

    /**
     * Returns true if this store has been parameterized as 'default'.
     */
    public boolean isDefaultStore ()
    {
        return MapUtils.getAsBoolean(getParams(), P_DEFAULT, false);
    }

    /**
     * Returns the flowExpressionId instances of the workitems belonging
     * to the given workflow instance.
     */
    public java.util.List findFlowInstance
        (final Subject s, final String workflowInstanceId)
    throws 
        StoreException
    {
        log.debug("findFlowInstance()");

        final int limit = 1500;
        final java.util.List result = new java.util.ArrayList(10);

        final java.util.Iterator it = 
            this.getStrategy.listWorkItems(s, limit).iterator();
        while (it.hasNext())
        {
            final InFlowWorkItem item = (InFlowWorkItem)it.next();
            final FlowExpressionId fei = item.getLastExpressionId();

            if (fei.getWorkflowInstanceId().equals(workflowInstanceId))
                result.add(item.getLastExpressionId());
        }

        return result;
    }

    /**
     * Returns a list of headers for this store.
     * A header is a 'summary' of a workitem, it's built by a HeaderFactory.
     * For each store, a different header factory may be used, giving a
     * specific view on the workitems contained in the store.
     */
    public java.util.List getHeaders 
        (final Subject s, final int limit)
    throws 
        StoreException
    {
        return getHeaders(s, limit, null);
    }

    /**
     * Returns a list of headers for this store, a comparator passed as
     * argument takes care of sorting the headers found.
     * A header is a 'summary' of a workitem, it's built by a HeaderFactory.
     * For each store, a different header factory may be used, giving a
     * specific view on the workitems contained in the store.
     */
    public java.util.List getHeaders 
        (final Subject s, 
         final int limit, 
         java.util.Comparator headerComparator)
    throws 
        StoreException
    {
        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "browse"));

        if (headerComparator == null)
            headerComparator = LastModifiedHeaderComparator.YOUNGER_FIRST;

        final long startTime = System.currentTimeMillis();

        int capacity = limit;
        if (capacity < 0) capacity = 100;

        final java.util.List result = new java.util.ArrayList(capacity);

        final java.util.List items = this.getStrategy.listWorkItems(s, limit);

        if (log.isDebugEnabled())
        {
            log.debug
                ("getHeaders() '"+getName()+
                 "' found "+items.size()+" workitems");
        }

        final java.util.Iterator it = items.iterator();
        while (it.hasNext())
        {
            final InFlowWorkItem wi = (InFlowWorkItem)it.next();

            //log.debug("loadHeaders() Building header...");

            try
            {
                final Header header = getHeaderFactory()
                    //.buildHeader(wi, isLocked(wi.getLastExpressionId()));
                    .buildHeader(wi, getLockerName(wi.getLastExpressionId()));

                result.add(header);
            }
            catch (final WorkListException wle)
            {
                throw new StoreException
                    ("Failed to loadHeaders from store '"+getName()+"'", wle);
            }
        }
        //log.debug("loadHeaders() Built headers");

        ((java.util.ArrayList)result).trimToSize(); 
            // less bandwidth required

        java.util.Collections.sort(result, headerComparator);

        if (log.isDebugEnabled())
        {
            log.debug
                ("getHeaders() '"+getName()+"' took "+
                 (System.currentTimeMillis()-startTime)+" ms");
        }

        return result;
    }

    /**
     * Will return true if the given subject has a lock on the workitem
     * designated by its lastExpressionId.
     */
    public boolean hasLockOn (Subject s, FlowExpressionId expressionId)
    {
        if (log.isDebugEnabled())
            log.debug(getName()+" hasLockOn() "+expressionId);

        final Lock lock = (Lock)this.lockMap.get(expressionId);

        if (log.isDebugEnabled())
            log.debug(getName()+" hasLockOn() found lock "+lock);

        if (lock == null) return false;

        //log.debug("hasLockOn() subject  >"+s+"<");
        //log.debug("hasLockOn() locker   >"+lock.getLocker()+"<");

        if ( ! s.equals(lock.getLocker())) return false;

        lock.touch(); 
            // ensure that the dispatching (that happens next) will be ok

        return true;
    }

    /**
     * Returns true if the store accepts workitems for the given
     * participant.
     */
    public boolean acceptsWorkItemsFor (final String participantName)
    {
        if (log.isDebugEnabled())
        {
            log.debug
                ("acceptsWorkItemsFor() (store '"+getName()+
                 "') ? >"+participantName+"<");
        }

        for (int i=0; i<this.acceptedParticipants.length; i++)
        {
            if (log.isDebugEnabled())
            {
                log.debug
                    ("acceptsWorkItemsFor() does '"+participantName+
                     "' match '"+this.acceptedParticipants[i]+"' ?");
            }

            if (participantName.matches(this.acceptedParticipants[i]))
            {
                log.debug("acceptsWorkItemsFor() true.");
                return true;
            }

            log.debug("acceptsWorkItemsFor() false");
        }

        log.debug("acceptsWorkItemsFor() no success : returning false.");

        return false;
    }

    /**
     * This method will return true if the current user/codebase has the
     * right to perform the given action on this store.
     */
    public boolean checkPermission (final String action)
    {
        if (log.isDebugEnabled())
        {
            log.debug
                ("checkPermission() checking '"+getName()+
                 "' for permission '"+action+"'");
        }

        try
        {
            java.security.AccessController.checkPermission
                (StorePermission.newStorePermission(getName(), action));
        }
        catch (final java.security.AccessControlException ace)
        {
            log.debug("checkPermission() returning false : permission denied");
            //log.debug("checkPermission() returning false", ace);
            return false;
        }
        log.debug("checkPermission() returning true");
        return true;
    }

    //
    // METHODS from Service

    /**
     * Stops this store (takes especially care of stopping the unlock daemon).
     */
    public void stop ()
        throws ServiceException
    {
        super.stop();

        this.unlockDaemon.cancel();
        log.info("stop() UnlockDaemon stopped.");

        log.info("stop() Service '"+getName()+"' stopped.");
    }

    //
    // STATIC METHODS

    // 
    // copied from openwfe.org.worklist.impl.AbstractWorkItemStore :
    //
    // Tue Oct 29 09:01:21 CET 2002
    // je suis dans un wagon restaurant 'Blaise Cendrars'. Est-ce
    // l'unique wagon-restaurant ddi  cet crivain ?
    //
    // "Wagon-restaurant. On ne distingue rien dehors. 
    // Il fait nuit noire."
    // Blaise Cendras (aka Frdric Sauser)
    //
    // Fait plaisir de croiser ce Suisse-l.
    //
    // Envie de relire le 'transsibrien'
    //

    //
    // INNER CLASSES

    /**
     * The unlock daemon wakes up from time to time to unlock workitem whose
     * lock has timed out.
     */
    class UnlockDaemon extends java.util.TimerTask
    {
        private final org.apache.log4j.Logger log = 
            org.apache.log4j.Logger.getLogger(UnlockDaemon.class.getName());

        public void run ()
        {
            //
            // golden rule :
            // always enclose those run() bodies inside a 'try catch throwable'
            //
            
            try
            {
                if (log.isDebugEnabled())
                {
                    log.debug
                        ("run() waking up for "+
                         SimpleWorkItemStore.this.getName());
                }

                synchronized (SimpleWorkItemStore.this.lockMap)
                {
                    final java.util.List toRemove = new java.util.ArrayList();

                    java.util.Iterator it = SimpleWorkItemStore.this.lockMap
                        .keySet().iterator();
                    while (it.hasNext())
                    {
                        final FlowExpressionId itemId = 
                            (FlowExpressionId)it.next();

                        final Lock lock = 
                            (Lock)SimpleWorkItemStore.this.lockMap.get(itemId);

                        if (lock.shouldBeUnlocked()) toRemove.add(itemId);
                    }

                    it = toRemove.iterator();
                    while (it.hasNext())
                    {
                        final FlowExpressionId itemId = 
                            (FlowExpressionId)it.next();
                        try
                        {
                            SimpleWorkItemStore.this.unlock(itemId);
                        }
                        catch (final StoreException se)
                        {
                            log.warn
                                ("run() unlock problem... resuming anyway...", 
                                 se);
                        }

                        if (log.isDebugEnabled())
                        {
                            log.debug
                                ("run() in "+SimpleWorkItemStore.this.getName()+
                                 ", unlocked and uncached "+itemId);
                        }
                    }
                }

                log.debug("run() going to sleep.");
            }
            catch (final Throwable t)
            {
                log.warn("UnlockDaemon : problem", t);
            }
        }
    }

}
