/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: QueryGetStrategy.java 3118 2006-08-30 14:38:36Z jmettraux $
 */

//
// DefaultGetStrategy.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.worklist.impl.store;

import javax.security.auth.Subject;

import openwfe.org.MapUtils;
import openwfe.org.FileUtils;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.auth.BasicPrincipal;
import openwfe.org.query.QueryMap;
import openwfe.org.query.QueryException;
import openwfe.org.query.item.QueryItem;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.worklist.store.GetStrategy;
import openwfe.org.worklist.store.StoreException;
import openwfe.org.worklist.store.WorkItemStore;


/**
 * A get strategy that looks up a query-map and produces a 'view' that
 * is potentially different for each subject.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: QueryGetStrategy.java 3118 2006-08-30 14:38:36Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class QueryGetStrategy

    extends DefaultGetStrategy

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(QueryGetStrategy.class.getName());

    //
    // CONSTANTS & co

    /**
     * The 'queryMap' parameter points to a query map service.
     */
    public final static String P_QUERY_MAP
        = "queryMap";

    /**
     * By default this strategy, uses a query map named 'queryMap'.
     */
    public final static String DEFAULT_QUERY_MAP
        = "queryMap";

    /**
     * This parameter is used to indicate where from the queries configuration
     * should be loaded.
     */
    public final static String P_QUERY_MAP_URL
        = "queryMapUrl";

    /**
     * is actually set to "file:./etc-worklist/query-map.xml".
     */
    public final static String DEFAULT_QUERY_MAP_URL
        //= "file:./etc/worklist/query-map.xml";
        = "etc/worklist/query-map.xml";

    //
    // FIELDS

    private String queryMapName = null;
    private java.net.URL queryMapUrl = null;

    private int factor = 2;

    //
    // CONSTRUCTORS

    /**
     * Performs the initialization of the strategy (give it enough
     * info to do its task).
     */
    public void init 
        (final ApplicationContext context, 
         final WorkItemStore store, 
         final java.util.Map storeParams,
         final String storageName)
    throws
        StoreException
    {
        super.init(context, store, storeParams, storageName);
        
        this.queryMapName = MapUtils.getAsString
            (getStoreParams(),
             P_QUERY_MAP,
             DEFAULT_QUERY_MAP);

        log.info("init() using queryMap named '"+this.queryMapName+"'");

        //
        // queryMapUrl
        
        final String sUrl = MapUtils.getAsString
            (getStoreParams(),
             P_QUERY_MAP_URL,
             DEFAULT_QUERY_MAP_URL);

        log.debug("init() sUrl is >"+sUrl+"<");

        try
        {
            this.queryMapUrl = FileUtils.toUrl(sUrl);
        }
        catch (final java.net.MalformedURLException mue)
        {
            throw new StoreException
                ("Could not resolve query-map at "+sUrl, mue);
        }

        log.info("init() using query-map at "+this.queryMapUrl.toString());

        //
        // check if queryMap is available
        
        if (getQueryMap() == null)
        {
            log.warn
                ("No query map found as service '"+this.queryMapName+"'...");
        }
    }

    //
    // METHODS

    /**
     * Returns the name of the query map that this query strategy uses.
     */
    public String getQueryMapName ()
    {
        return this.queryMapName;
    }

    /**
     * Sets the query map that this query strategy uses.
     */
    public void setQueryMapName (final String s)
    {
        this.queryMapName = s;
    }

    /**
     * Returns the query map instance that this query strategy uses 
     * (fetches it from the context).
     */
    public QueryMap getQueryMap ()
    {
        return (QueryMap)getContext().get(this.queryMapName);
    }

    //
    // METHODS from GetStrategy

    public java.util.List listWorkItems (final Subject s, final int limit)
        throws StoreException
    {
        //
        // prepare query
        
        final StringBuffer sb = new StringBuffer(this.queryMapUrl.toString());
        sb.append("#");
        sb.append(getStoreName());
        sb.append("#");
        sb.append(BasicPrincipal.getBasicPrincipal(s).getName());

        final String query = sb.toString();
        
        QueryItem queryItem = null;
        try
        {
            queryItem = getQueryMap().lookupQuery(query);
        }
        catch (final QueryException qe)
        {
            throw new StoreException
                ("Couldn't instantiate query "+query, qe);
        }

        if (queryItem == null)
        {
            //throw new StoreException
            //    ("Query named '"+query+"' could not be found");
            log.info
                ("Query named '"+query+"' could not be found");

            return new java.util.ArrayList(0);
        }
        
        if (log.isDebugEnabled())
        {
            log.debug
                ("listWorkItems() found query of class '"+
                 queryItem.getClass().getName()+"'");
        }

        //
        // do the query

        int sLimit = limit * this.factor;
        int targetSize = limit;
        if (limit < 0)
        {
            sLimit = -1;
            targetSize = 100;
        }

        final java.util.List items = super.listWorkItems(s, sLimit);

        if (items.size() < targetSize) targetSize = items.size();

        final java.util.List results = new java.util.ArrayList(targetSize);

        try
        {
            final java.util.Iterator it = items.iterator();
            while (it.hasNext())
            {
                final InFlowWorkItem item = (InFlowWorkItem)it.next();

                if (queryItem.validate(item)) results.add(item);
            }
        }
        catch (final QueryException qe)
        {
            log.debug
                ("Query (validation) failure", qe);
            throw new StoreException
                ("Query (validation) failure", qe);
        }

        if (log.isDebugEnabled())
        {
            log.debug
                ("listWorkItems() matching workitems : "+results.size()+
                 " / "+items.size());
        }

        return results;
    }

    /**
     * Returns the count of workitems (satisfying the query strategy 
     * criteria).
     */
    public int countWorkItems 
        (final Subject s)
    throws 
        StoreException
    {
        return listWorkItems(s, -1).size();
    }

    //
    // STATIC METHODS

}
