/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: InMemoryWorkItemStorage.java 3118 2006-08-30 14:38:36Z jmettraux $
 */

//
// InMemoryWorkItemStorage.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.worklist.impl.store;

import openwfe.org.Utils;
import openwfe.org.MapUtils;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.xml.XmlUtils;
import openwfe.org.misc.DirectoryFilter;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.worklist.store.StoreException;


/**
 * An implementation of WorkItemStorage that stores workitems in files.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: InMemoryWorkItemStorage.java 3118 2006-08-30 14:38:36Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class InMemoryWorkItemStorage

    extends AbstractStorage
    
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(InMemoryWorkItemStorage.class.getName());

    //
    // CONSTANTS & co

    //
    // FIELDS

    private java.util.Map storeMap = null;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        //
        // init stores

        this.storeMap = new java.util.HashMap();

        //
        // done

        log.info("init() storage '"+serviceName+"' ready.");
    }

    //
    // METHODS

    /**
     * Returns the store entry for the given store.
     * Will throw a StoreException if the store is not present.
     */
    protected StoreEntry getStore (final String storeName)
        throws StoreException
    {
        StoreEntry result = (StoreEntry)this.storeMap.get(storeName);

        if (result != null) return result;

        throw new StoreException
            ("There is no store named '"+storeName+"' here.");
    }

    /**
     * Returns the store entry for the given store.
     * Will create a new store if there is currently no store for the
     * given store name.
     */
    protected synchronized StoreEntry doGetStore (final String storeName)
    {
        StoreEntry result = (StoreEntry)this.storeMap.get(storeName);

        if (result != null) return result;

        result = new StoreEntry(storeName);

        this.storeMap.put(storeName, result);

        return result;
    }

    //
    // METHODS from WorkItemStorage

    /**
     * Inserts a workitem into the storage for given store.
     */
    public void storeWorkItem 
        (final String storeName, final InFlowWorkItem wi)
    throws 
        StoreException
    {
        doGetStore(storeName).store(wi);
    }

    /**
     * Removes the workitem from the storage for a given store.
     */
    public void removeWorkItem 
        (final String storeName, final FlowExpressionId fei)
    throws 
        StoreException
    {
        getStore(storeName).remove(fei);
    }

    /**
     * Retrieves a workitem given the storename and the workitem id.
     */
    public InFlowWorkItem retrieveWorkItem 
        (final String storeName, final FlowExpressionId fei)
    throws 
        StoreException
    {
        final StoreEntry store = (StoreEntry)this.storeMap.get(storeName);
        if (store == null) return null;

        return store.get(fei);
    }

    /**
     * Returns the number of workitems in a store.
     */
    public int countWorkItems 
        (final String storeName)
    throws 
        StoreException
    {
        final StoreEntry store = (StoreEntry)this.storeMap.get(storeName);
        if (store == null) return 0;

        return store.size();
    }

    /**
     * Returns a list of workitems from a store.
     */
    public java.util.List listWorkItems 
        (final String storeName, final int limit)
    throws 
        StoreException
    {
        return getStore(storeName).list();
    }

    //
    // METHODS from Service

    /**
     * Status is outputted as a JDOM element. The status is various
     * information about a service activities and state.
     */
    public org.jdom.Element getStatus ()
    {
        org.jdom.Element result = new org.jdom.Element(getName());

        result.addContent(XmlUtils.getClassElt(this));
        result.addContent(XmlUtils.getRevisionElt("$Id: InMemoryWorkItemStorage.java 3118 2006-08-30 14:38:36Z jmettraux $"));


        final StoreStat stat = new StoreStat();

        final java.util.Iterator it = this.storeMap.keySet().iterator();

        while (it.hasNext())
        {
            final String storeName = (String)it.next();

            final java.util.Iterator jit = 
                ((StoreEntry)this.storeMap.get(storeName)).list().iterator();
            while (jit.hasNext())
            {
                final InFlowWorkItem wi = (InFlowWorkItem)jit.next();

                stat.put
                    (storeName, 
                     wi.getLastExpressionId().getWorkflowDefinitionUrl(),
                     wi.getLastExpressionId().getParentWorkflowInstanceId());
            }
        }

        return result;
    }

    //
    // STATIC METHODS

    //
    // INNER CLASSES

    private static class StoreEntry 
    {
        private String name = null;
        private java.util.Map map = null;
        private java.util.List list = null;

        public StoreEntry (final String storeName)
        {
            super();

            this.name = storeName;

            this.map = new java.util.HashMap();
            this.list = new java.util.LinkedList();
        }

        public void store (final InFlowWorkItem wi)
        {
            synchronized (this)
            {
                this.map.put(wi.getLastExpressionId(), wi);
                this.list.add(wi);
            }
        }

        public InFlowWorkItem get (final FlowExpressionId fei)
        {
            return (InFlowWorkItem)this.map.get(fei);
        }

        public void remove (final FlowExpressionId fei)
        {
            synchronized (this)
            {
                final InFlowWorkItem wi = (InFlowWorkItem)this.map.remove(fei);

                if (wi != null) 
                    this.list.remove(wi);
            }
        }

        public java.util.Iterator iterator ()
        {
            return this.list.iterator();
        }

        public int size ()
        {
            return this.list.size();
        }

        public java.util.List list ()
        {
            return new java.util.ArrayList(this.list);
        }
    }

}
