/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: FileWorkItemStorage.java 3118 2006-08-30 14:38:36Z jmettraux $
 */

//
// FileWorkItemStorage.java
//
// john.mettraux@openwfe.org
//
//
// contains a bug fix by Olivier Stembert
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.worklist.impl.store;

import openwfe.org.MapUtils;
import openwfe.org.FileUtils;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.xml.XmlUtils;
import openwfe.org.misc.DirectoryFilter;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.worklist.store.StoreException;


/**
 * An implementation of WorkItemStorage that stores workitems in files.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: FileWorkItemStorage.java 3118 2006-08-30 14:38:36Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class FileWorkItemStorage

    extends AbstractStorage
    
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(FileWorkItemStorage.class.getName());

    //
    // CONSTANTS & co

    /**
     * The parameter 'directory' is used to tell the FileWorkItemStore where
     * its root is located.
     */
    public final static String P_DIRECTORY
        = "directory";

    /**
     * The default work directory for this storage is work/storage/
     */
    public final static String DEFAULT_DIRECTORY
        = "work"+java.io.File.separator+"storage";

    private final static java.io.FilenameFilter fileNameFilter =
        new java.io.FilenameFilter ()
        {
            public boolean accept (java.io.File dir, String fileName)
            {
                return fileName.endsWith(".xml");
            }
        };

    //
    // FIELDS

    private String directory = null;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        //
        // determine work directory

        this.directory = MapUtils.getAsString
            (getParams(),
             P_DIRECTORY,
             DEFAULT_DIRECTORY);
        
        this.directory = FileUtils.getCanonicalPath
            (getContext().getApplicationDirectory(), this.directory);

        if ( ! this.directory.endsWith(java.io.File.separator))
            this.directory += java.io.File.separator;

        log.info("init() storage directory is "+this.directory);

        //
        // done
        
        log.info("init() storage '"+serviceName+"' ready.");
    }

    //
    // METHODS

    /**
     * Returns the path where this store puts its workitems.
     */
    protected String getDirectory () 
    {
        return this.directory;
    }

    /**
     * Given a workitem id, tells under which file this workitem should
     * get saved.
     */
    protected String determineFileName 
        (final String storeName, 
         final FlowExpressionId fei,
         final boolean shouldMkdirs)
    {
        if (shouldMkdirs)
        {
            final java.io.File f = new java.io.File(this.directory+storeName);
            if ( ! f.exists()) f.mkdirs();

            if (log.isDebugEnabled())
                log.debug("determineFileName() built dir "+f.getPath());
        }

        StringBuffer sb = new StringBuffer();

        sb.append(this.directory);
        sb.append(storeName);
        sb.append(java.io.File.separatorChar);

        sb.append(determineFileName(fei));

        return sb.toString();
    }

    /**
     * As this method is also used by the AppliedWorkitemExtractor tool,
     * it's been made public (and static).
     */
    public static String determineFileName (final FlowExpressionId fei)
    {
        final StringBuffer sb = new StringBuffer();

        sb.append(FileUtils.ensureForFileName(fei.getInitialEngineId()));
        sb.append("--");
        sb.append(fei.getWorkflowDefinitionName());
        sb.append("--");
        sb.append(fei.getWorkflowDefinitionRevision());
        sb.append("--");
        sb.append(fei.getWorkflowInstanceId());
        sb.append("--");
        sb.append(fei.getExpressionId());
        sb.append(".xml");

        return sb.toString();
    }

    //
    // METHODS from WorkItemStorage

    /**
     * Inserts a workitem into the storage for given store.
     */
    public void storeWorkItem 
        (final String storeName, final InFlowWorkItem wi)
    throws 
        StoreException
    {
        final FlowExpressionId id = wi.getLastExpressionId();
        final String fileName = determineFileName(storeName, id, true);

        java.io.FileOutputStream fos = null;
        try
        {
            final byte[] encodedWorkItem = (byte[])getWorkItemCoder()
                .encode(wi, getContext(), getParams());

            fos = new java.io.FileOutputStream(fileName);

            fos.write(encodedWorkItem);
            fos.flush();
        }
        catch (Exception e)
        {
            throw new StoreException
                ("Failed to store workitem in file '"+fileName+"'", e);
        }
        finally
        {
            try
            {
                fos.close();
            }
            catch (final Throwable t)
            {
                // ignore
            }
        }

        if (log.isDebugEnabled())
            log.debug("storeWorkItem() saved wi in "+fileName);
    }

    /**
     * Removes the workitem from the storage for a given store.
     */
    public void removeWorkItem 
        (final String storeName, final FlowExpressionId fei)
    throws 
        StoreException
    {
        final String fileName = determineFileName(storeName, fei, false);
        final java.io.File file = new java.io.File(fileName);

        boolean deleted = file.delete();

        if ( ! log.isDebugEnabled()) return;

        if ( ! deleted)
            log.debug("removeWorkItem() failed to remove "+fileName);
        else
            log.debug("removeWorkItem() immediately deleted "+fileName);
    }

    /**
     * Retrieves a workitem given the storename and the workitem id.
     */
    public InFlowWorkItem retrieveWorkItem 
        (final String storeName, final FlowExpressionId fei)
    throws 
        StoreException
    {
        InFlowWorkItem wi = null;

        String fileName = determineFileName(storeName, fei, false);

        fileName = FileUtils.getCanonicalPath
            (getContext().getApplicationDirectory(), fileName);

        java.io.FileInputStream fis = null;
        try
        {
            fis = new java.io.FileInputStream(fileName);

            wi = (InFlowWorkItem)getWorkItemCoder().decode
                (fis,
                 getContext(),
                 getParams());
        }
        catch (final Throwable t)
        {
            log.warn
                ("retrieveWorkItem() "+
                 "failed to load workitem '"+fei+
                 "' from '"+fileName+"'", t);

            throw new StoreException
                ("failed to load workitem '"+fei+
                 "' from '"+fileName+"'", t);
        }
        finally
        {
            try
            {
                fis.close();
            }
            catch (final Throwable t)
            {
                // ignore
            }
        }

        if (wi == null)
        {
            throw new StoreException
                ("Empty workitem '"+fei+"' in '"+fileName+"'");
        }

        return wi;
    }

    /**
     * Returns the number of workitems in a store.
     */
    public int countWorkItems 
        (final String storeName)
    throws 
        StoreException
    {
        final java.io.File dir = new java.io.File
            (this.directory + storeName);

        if (dir == null) 
        {
            //throw new StoreException
            log.warn
                ("directory "+this.directory+storeName+
                 " seems not to exist. "+
                 "it is required by store '"+storeName+"'");
            return 0;
        }

        final java.io.File[] files = dir.listFiles(fileNameFilter);

        if (files == null) 
        {
            //throw new StoreException
            log.warn
                ("countWorkItems() "+
                 "IO error while counting files in directory "+this.directory+
                 " for store '"+storeName+
                 "' (or perhaps directory is not a directory).");
            return 0;
        }

        return files.length;
    }

    /**
     * Returns a list of workitems from a store.
     */
    public java.util.List listWorkItems 
        (final String storeName, final int limit)
    throws 
        StoreException
    {
        final java.io.File dir = new java.io.File
            (this.directory + storeName);

        if (log.isDebugEnabled())
        {
            log.debug("listWorkItems() dir is "+dir.getPath());
            log.debug("listWorkItems("+limit+");");
        }

        //
        // do the job

        java.io.File[] files = dir.listFiles(fileNameFilter);

        if (files == null) 
        {
            log.debug("listWorkItems() found no files.");
            return new java.util.ArrayList(0);
        }

        if (log.isDebugEnabled())
            log.debug("listWorkItems() files to consider : "+files.length);

        java.util.List result = null;

        if (limit > 0)
            result = new java.util.ArrayList(limit);
        else
            result = new java.util.ArrayList();

        for (int i=0; i<files.length; i++)
        {
            if (limit > 0 && i >= limit) break;

            if (log.isDebugEnabled())
                log.debug("listWorkItems() considering "+files[i]);

            java.io.FileInputStream fis = null;
            try
            {
                fis = new java.io.FileInputStream(files[i].getPath());

                final InFlowWorkItem wi = (InFlowWorkItem)getWorkItemCoder()
                    .decode(fis, getContext(), getParams());

                result.add(wi);
            }
            catch (final Exception e)
            {
                // ignore this workitem
                log.debug("GASP!!!!", e);
                log.debug("Failed to load workitem in "+files[i].getPath());
            }
            finally
            {
                try
                {
                    fis.close();
                }
                catch (Throwable t)
                {
                    // ignore
                }
            }
        }

        if (log.isDebugEnabled())
            log.debug("listWorkItems() found "+result.size()+" workitems");

        return result;
    }

    //
    // METHODS from Service

    /**
     * Status is outputted as a JDOM element. The status is various
     * information about a service activities and state.
     */
    public org.jdom.Element getStatus ()
    {
        org.jdom.Element result = new org.jdom.Element(getName());

        result.addContent(XmlUtils.getClassElt(this));
        result.addContent(XmlUtils.getRevisionElt("$Id: FileWorkItemStorage.java 3118 2006-08-30 14:38:36Z jmettraux $"));

        //
        // list file names to determine storage content
        
        java.io.File dir = new java.io.File(this.directory);

        java.io.File[] dirs = dir.listFiles(new DirectoryFilter());

        final StoreStat stat = new StoreStat();

        if (dirs != null)
        {
            for (int j=0; j<dirs.length; j++)
            {
                if (log.isDebugEnabled())
                    log.debug("getStatus() dirs["+j+"] is "+dirs[j]);

                java.io.File[] files = dirs[j].listFiles(fileNameFilter);
                if (files != null)
                {
                    final String storeName = dirs[j].getName();

                    for (int i=0; i<files.length; i++)
                    {
                        String fn = files[i].getName();

                        if (log.isDebugEnabled())
                            log.debug("getStatus() file is "+fn);

                        fn = fn.substring(0, fn.length()-4); // removing .xml
                        String[] ss = fn.split("--");

                        String flowDef = 
                            ss[0]+" "+ss[1];

                        String flowId = 
                            flowDef+" "+FlowExpressionId.extractParentWorkflowInstanceId(ss[2]);

                        stat.put(storeName, flowDef, flowId);
                    }
                }
            }
        }

        result.addContent(stat.render());

        //
        // the end

        return result;
    }

    //
    // STATIC METHODS

}
