/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: CorrelatedPutStrategy.java 3334 2006-09-17 06:19:18Z jmettraux $
 */

//
// DefaultPutStrategy.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.worklist.impl.store;

import openwfe.org.MapUtils;
import openwfe.org.ApplicationContext;
import openwfe.org.util.beancoder.XmlBeanCoder;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.workitem.Attribute;
import openwfe.org.engine.workitem.StringAttribute;
import openwfe.org.engine.workitem.StringMapAttribute;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.engine.expressions.sync.MergeUtils;
import openwfe.org.engine.participants.Participant;
import openwfe.org.worklist.store.Lock;
import openwfe.org.worklist.store.PutStrategy;
import openwfe.org.worklist.store.WorkItemStore;
import openwfe.org.worklist.store.StoreException;


/**
 * This implementation takes care of correlating workitems :
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: CorrelatedPutStrategy.java 3334 2006-09-17 06:19:18Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class CorrelatedPutStrategy

    extends AbstractStoreStrategy

    implements PutStrategy

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(CorrelatedPutStrategy.class.getName());

    //
    // CONSTANTS & co
    
    /**
     * The parameter 'cacheFileName' indicates to this strategy where it should 
     * save its state.
     */
    public final static String P_CACHE_FILE_NAME
        = "correlationCacheFileName";

    private final static String DEFAULT_CACHE_FILE_NAME
        = openwfe.org.Utils.getTempDir()+"/"+
          CorrelatedPutStrategy.class.getName()+"__cache.xml";

    /**
     * The parameter 'correlationSlaveStoreName' tells the strategy where
     * to store the pending slave workitems (the ones waiting to get
     * inserted in their master workitem).
     * If this parameter is not specified, the slave store name will be the
     * regular storename suffixed with "_cslaves".
     */
    public final static String P_SLAVE_STORE_NAME
        = "correlationSlaveStoreName";

    /**
     * '__correlation_id__' : a workitem with such an attribute is said to be 
     * correlated and belongs to the correlation class as other workitems
     * with the same value (a StringAttribute) as correlation id.
     */
    public final static String CORRELATION_ID
        = "__correlation_id__";

    /**
     * '__correlation_field__' : a correlated workitem with this attribute set
     * is said to be a 'slave correlated workitem', a correlated workitem 
     * without this attribute is a 'master'. There can be only one master
     * per correlation class.
     */
    public final static String CORRELATION_FIELD
        = "__correlation_field__";

    /**
     * If the slave workitem has its attribute '__correlation_bounce__' set
     * to a string [attribute] value of 'true', this strategy will immediately
     * reply (forward) the slave workitem (after having kept a copy of it).
     * Else, the reply will occur only if (or when) the master workitem has
     * arrived / arrives.
     */
    public final static String CORRELATION_BOUNCE
        = "__correlation_bounce__";

    //
    // FIELDS

    private String cacheFileName = null;

    /* keeping track of the master workitems' FlowExpressionId */
    private java.util.Map masters = null;

    /* storing the slaves whose master has not yet arrived */
    private java.util.Map slaves = null;

    private String slaveStoreName = null;

    //
    // CONSTRUCTORS

    /**
     * Performs the initialization of the strategy (give it enough
     * info to do its task).
     */
    public void init 
        (final ApplicationContext context, 
         final WorkItemStore store, 
         final java.util.Map storeParams,
         final String storageName)
    throws
        StoreException
    {
        super.init(context, store, storeParams, storageName);

        this.cacheFileName = MapUtils.getAsString
            (storeParams,
             P_CACHE_FILE_NAME,
             DEFAULT_CACHE_FILE_NAME);

        log.debug("init() using cache at "+this.cacheFileName);

        this.slaveStoreName = MapUtils.getAsString
            (storeParams,
             P_SLAVE_STORE_NAME,
             store.getName()+"_cslaves");

        log.debug("init() slaveStoreName set to "+this.slaveStoreName);

        loadState();
    }

    //
    // METHODS

    private void loadState ()
    {
        final java.io.File f = new java.io.File(this.cacheFileName);

        if ( ! f.exists()) 
        {
            log.debug
                ("loadState() nothing to load, no file at "+this.cacheFileName);

            this.masters = new java.util.HashMap();
            this.slaves = new java.util.HashMap();

            return;
        }

        try
        {
            final java.util.Map state = 
                (java.util.Map)XmlBeanCoder.load(this.cacheFileName);

            this.masters = (java.util.Map)state.get("masters");
            this.slaves = (java.util.Map)state.get("slaves");

            log.debug("loadState() done.");
        }
        catch (final Exception e)
        {
            log.warn("loadState() failure with file "+this.cacheFileName, e);
        }
    }

    private void saveState ()
    {
        log.debug("saveState()");

        final java.util.Map state = new java.util.HashMap(2);
        state.put("masters", this.masters);
        state.put("slaves", this.slaves);

        try
        {
            XmlBeanCoder.save(this.cacheFileName, state);
        }
        catch (final Exception e)
        {
            log.warn
                ("saveState() failed to save state. "+
                 "Will lose it at system shutdown.", e);
        }
    }

    //
    // METHODS from PutStrategy

    /**
     * Checks wether the incoming workitem is correlated (slave or master) or
     * not correlated at all, then takes the appropriate action.
     */
    public void put (final InFlowWorkItem wi)
        throws StoreException
    {
        final String cId = wi.getAttributes().sget(CORRELATION_ID);

        if (cId != null)
        {
            //log.debug("put() wi with correlationId (set to >"+cId+"<)");

            final String cField = wi.getAttributes().sget(CORRELATION_FIELD);

            synchronized (this)
            {
                if (cField != null)
                    receiveSlave(cId, cField, wi);
                else
                    receiveMaster(cId, wi);
            }

            return;
        }

        //
        // no correlation, behaving like a regular put strategy...

        getStorage().storeWorkItem(getStoreName(), wi);
    }

    private void receiveSlave 
        (final String cId, final String cField, final InFlowWorkItem wi)
    throws 
        StoreException
    {
        log.debug("receiveSlave() for '"+cId+"'  (field : '"+cField+"')");

        final FlowExpressionId masterId = 
            (FlowExpressionId)this.masters.get(cId);

        InFlowWorkItem masterWi = null;
        if (masterId != null)
        {
            try
            {
                masterWi = getStore().getAndLock(null, masterId);
            }
            catch (final StoreException se)
            {
                this.masters.remove(cId);
                log.debug("receiveSlave() did not find master");
            }
        }

        if (masterId == null || masterWi == null)
            //
            // master hasn't arrived yet
        {
            log.debug("receiveSlave() master hasn't arrived yet.");

            getStorage().storeWorkItem(this.slaveStoreName, wi);

            putSlave(cId, cField, wi.getLastExpressionId());
            saveState();

            if (shouldBounceDirectly(wi)) asyncBounce(wi);

            return;
        }

        log.debug("receiveSlave() master has already arrived");

        //
        // master already found

        merge(cField, masterWi, wi);

        try
        {
            getStorage().storeWorkItem(getStoreName(), masterWi);
        }
        catch (final StoreException se)
        {
            throw new StoreException
                ("Failed to re-store master after insertion of "+
                 "correlated slave's attributes", se);
        }

        //
        // release master workitem

        getStore().release(null, masterId);

        //
        // bounce asynchronously
        
        asyncBounce(wi);
    }

    private void putSlave 
        (final String correlId, 
         final String correlField, 
         final FlowExpressionId slaveWorkitemId)
    {
        java.util.List sList = (java.util.List)this.slaves.get(correlId);

        if (sList == null)
        {
            sList = new java.util.ArrayList(7);
            this.slaves.put(correlId, sList);
        }

        sList.add(new SlaveEntry(correlField, slaveWorkitemId));
    }

    private void receiveMaster 
        (final String cId, InFlowWorkItem wi)
    throws
        StoreException
    {
        log.debug("receiveMaster() for '"+cId+"'");

        //if (this.masters.get(cId) != null)
        //{
        //    log.warn
        //      ("receiveMaster() there is already a master for id '"+cId+
        //       "'. Not registering later one as master.");
        //    return;
        //}
        if (this.masters.get(cId) != null)
        {
            log.debug
                ("receiveMaster() "+
                 "already had a master for "+cId+". merging...");

            wi = mergeMaster(wi);
        }
        else
        {
            this.masters.put(cId, wi.getLastExpressionId());
        }

        //
        // maybe there are pending slaves
       
        final java.util.List slaveIds = (java.util.List)this.slaves.get(cId);

        if (slaveIds != null)
        {
            log.debug("receiveMaster() slaves to consider : "+slaveIds.size());

            final java.util.Iterator it = slaveIds.iterator();
            while (it.hasNext())
            {
                final SlaveEntry se = (SlaveEntry)it.next();

                final InFlowWorkItem slaveWi = getStorage()
                    .retrieveWorkItem(this.slaveStoreName, se.workitemId);

                merge(se.cField, wi, slaveWi);

                getStorage()
                    .removeWorkItem(this.slaveStoreName, se.workitemId);

                // 
                // reply with slaveWi

                //if ( ! shouldBounceDirectly(slaveWi)) bounce(slaveWi);
                if ( ! shouldBounceDirectly(slaveWi)) asyncBounce(slaveWi);
            }

            this.slaves.remove(cId);

            saveState();
        }
        else
        {
            log.debug("receiveMaster() no slaves to consider.");
        }

        //
        // storing master

        getStorage().storeWorkItem(getStoreName(), wi);
    }

    private InFlowWorkItem mergeMaster 
        (final InFlowWorkItem incomingMaster)
    throws 
        StoreException
    {
        final InFlowWorkItem currentMaster = getStorage().retrieveWorkItem
            (getStore().getName(), incomingMaster.getLastExpressionId());

        //log.debug
        //    ("mergeMaster() 0 currentMaster.fieldCount :   "+
        //     currentMaster.getAttributes().size());
        //log.debug
        //    ("mergeMaster() 0 incomingMaster.fieldCount :  "+
        //     incomingMaster.getAttributes().size());

        //log.debug
        //    ("mergeMaster() 0 currentMaster :\n"+
        //     XmlBeanCoder.dumpToString(currentMaster));
        //log.debug
        //    ("mergeMaster() 0 incomingMaster :\n"+
        //     XmlBeanCoder.dumpToString(incomingMaster));

        MergeUtils.mergeInPlace(currentMaster, incomingMaster);

        //log.debug
        //    ("mergeMaster() 1 currentMaster.fieldCount :   "+
        //     currentMaster.getAttributes().size());
        //log.debug
        //    ("mergeMaster() 1 currentMaster :\n"+
        //     XmlBeanCoder.dumpToString(currentMaster));

        return currentMaster;
    }

    private void merge 
        (final String correlationField, 
         final InFlowWorkItem masterWi,
         final InFlowWorkItem slaveWi)
    {
        //
        // preparing the correlation field

        StringMapAttribute result = null;

        final Attribute a = masterWi.getAttributes().aget(correlationField);

        if (a != null && (a instanceof StringMapAttribute))
            result = (StringMapAttribute)a;
        else
            result = new StringMapAttribute();

        log.debug
            ("merge() "+
             "correlationField contained "+result.size()+" sub-fields");

        //
        // filling it

        final java.util.Iterator it = 
            slaveWi.getAttributes().keySet().iterator();
        while (it.hasNext())
        {
            final StringAttribute key = (StringAttribute)it.next();
            final String skey = key.toString();

            if (skey.equals(CORRELATION_ID) ||
                skey.equals(CORRELATION_FIELD) ||
                skey.equals(CORRELATION_BOUNCE))
            {
                continue;
            }

            result.put(key, (Attribute)slaveWi.getAttributes().get(key));
        }

        log.debug
            ("merge() "+
             "correlationField now contains "+result.size()+" sub-fields");

        masterWi.getAttributes().put(correlationField, result);
    }

    /*
     * Should this slave workitem be bounced back just after its arrival ?
     */
    private boolean shouldBounceDirectly (final InFlowWorkItem wi)
    {
        final String sBounce = wi.getAttributes().sget(CORRELATION_BOUNCE);

        if (sBounce == null) return false;

        return (sBounce.equalsIgnoreCase("true"));
    }

    /*
     * Forwards a workitem back to its emitting engine (participant expression)
     */
    private void bounce (final InFlowWorkItem wi)
        throws StoreException
    {
        log.debug("bounce() boucing back "+wi.getLastExpressionId());

        final String engineId = wi.getLastExpressionId().getEngineId();

        final Participant pEngine = openwfe.org.engine.Definitions
            .getParticipantMap
                (((openwfe.org.Service)getStore()).getContext())
            .get(engineId);

        try
        {
            pEngine.dispatch(getContext(), wi);
        }
        catch (final Exception e)
        {
            throw new StoreException
                ("Failed bounce slave correlated workitem", e);
        }

        log.debug("bounce() bounced "+wi.getLastExpressionId());
    }

    /*
     * bounce() but in its own thread
     */
    private void asyncBounce (final InFlowWorkItem wi)
    {
        (new Thread()
         {
             public void run ()
             {
                 try
                 {
                     bounce(wi);
                 }
                 catch (final StoreException se)
                 {
                     log.warn
                         ("Failed to bounce wi back  "+wi.getLastExpressionId(),
                          se);
                 }
             }
         }).start();
    }

    /**
     * Removes a workitem from the underlying storage.
     */
    public void remove (final FlowExpressionId id)
        throws StoreException
    {
        if (this.masters.containsValue(id))
        {
            final java.util.Iterator it = this.masters.keySet().iterator();
            while (it.hasNext())
            {
                final String correlationId = 
                    (String)it.next();
                final FlowExpressionId masterId = 
                    (FlowExpressionId)this.masters.get(correlationId);

                if (masterId.equals(id)) 
                {
                    this.masters.remove(correlationId);

                    log.debug
                        ("remove() removed master for correlationId '"+
                         correlationId+"'");

                    break;
                }
            }
        }

        getStorage().removeWorkItem(getStoreName(), id);
    }

    //
    // STATIC METHODS

    //
    // INNER CLASSES

    protected static class SlaveEntry
    {
        /* correlation field */
        private String cField = null;

        /* slave workitem id */
        private FlowExpressionId workitemId = null;

        public SlaveEntry 
            (final String cField, 
             final FlowExpressionId workitemId)
        {
            this.cField = cField;
            this.workitemId = workitemId;
        }

        public String getCorrelationField ()
        {
            return this.cField;
        }

        public FlowExpressionId getWorkitemId ()
        {
            return this.workitemId;
        }

        public void setCorrelationField (final String cField)
        {
            this.cField = cField;
        }

        public void setWorkitemId (final FlowExpressionId fei)
        {
            this.workitemId = fei;
        }
    }

}
